/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.datastructures.cache;

import java.util.concurrent.TimeUnit;

/**
 * A timed (Key, Value) cache which disposes the cached entry after the
 * specified amount of time if it is not released from the cache during that
 * time. The hold time is given to the {@link #put(Object, Object, long)}
 * method, separately for each value.
 * 
 * @author Tuukka Lehtonen
 * 
 * @see SoftTimedCache
 * @param <K> key type
 * @param <V> value type
 */
public interface ITimedCache<K, V> {

    /**
     * Put the value v into this cache with key k to be kept for holdTimeMs
     * milliseconds. After this times has elapsed, the (k,v) pair will be
     * disposed of and it will no longer be retrievable with key k.
     * 
     * @param k key with which the specified value is to be associated
     * @param v value to be associated with the specified key
     * @param holdTime the maximum amount of time units to hold on to the
     *        specified value
     * @param unit the unit of <code>holdTime</code>
     */
    void put(final K k, V v, long holdTime, TimeUnit unit);

    /**
     * Releases the value associated to the specified key k from this cache. If
     * the hold time for this particular (key, value) pair has ran out, the
     * value will no longer exist in the cache and <code>null</code> will be
     * returned.
     * 
     * @param k key of the value to release from the cache.
     * @return <code>null</code> if there is no value associated to the
     *         specified key
     */
    V release(K k);

}
