/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 *
 * @author Toni Kalajainen
 */
package org.simantics.utils.datastructures.cache;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.simantics.utils.datastructures.disposable.AbstractDisposable;

/**
 * CachedProvider provides values and stores the results in a cache.
 * Cached values must be explicitly released with clear, retain or remove.
 * Values are held with strong references.
 *
 * @param <K> the key
 * @param <V> the value
 */
public class CachedMapProvider<K, V> extends AbstractDisposable implements IMapProvider<K, V> {

	private Map<K, V> cache;
	
	private final IMapProvider<K, V> provider;
	
	/**
	 * Constructs new strong cache
	 * 
	 * @param provider provider of values
	 */
	public CachedMapProvider(IMapProvider<K, V> provider)
	{
		assert(provider!=null);
		this.provider = provider;
		this.cache = new HashMap<K, V>();
	}		

	/**
	 * Constructs new strong cache
	 * 
	 * @param provider provider of values
	 * @param cacheMap the map to use
	 */
	public CachedMapProvider(IMapProvider<K, V> provider, Map<K, V> cacheMap)
	{
		assert(provider!=null);
		this.provider = provider;
		this.cache = cacheMap;
	}		
	
	@Override
	public synchronized V get(K key) {
		assertNotDisposed();
		V value = cache.get(key);
		if (value!=null) return value;
		value = provider.get(key);
		assert(value!=null);
		cache.put(key, value);			
		return value;
	}
	
	public synchronized void clear()
	{
		cache.clear();
	}
	
	public synchronized void retain(Set<K> keys)
	{
		cache.keySet().retainAll(keys);
	}

	public synchronized void remove(Set<K> keys)
	{
		cache.keySet().removeAll(keys);
	}
	
	public synchronized void load(Set<K> keys)
	{
		for (K key : keys)
			get(key);
	}
	
	public synchronized void addAll(CachedMapProvider<K, V> anotherCache)
	{
		assertNotDisposed();
		cache.putAll(anotherCache.cache);
	}	
	
	public IMapProvider<K, V> getProvider() {
		return provider;
	}
	
	public synchronized Map<K, V> getAll()
	{
		assertNotDisposed();
		return new HashMap<K, V>(cache);
	}

	@Override
	protected void doDispose() {
		clear();
	}
	
}
