/*******************************************************************************
 * Copyright (c) 2007, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - gitlab #883 fix compareTo
 *******************************************************************************/
package org.simantics.utils.datastructures;


/**
 * A generic Pair (2-tuple) structure for containing two object instances of
 * chosen types.
 * 
 * @param <T1> type of first element
 * @param <T2> type of second element
 */
public final class Pair<T1, T2> implements Comparable<Pair<T1, T2>> {
    public final T1 first;
    public final T2 second;
    private final int hash;

    public static <T1, T2> Pair<T1, T2> make(T1 t1, T2 t2) {
        return new Pair<T1, T2>(t1, t2);
    }

    public Pair(T1 first, T2 second) {
        //assert(first != null);
        //assert(second != null);
        
        this.first = first;
        this.second = second;
        this.hash = makeHash();
    }
    
    @Override
    public boolean equals(Object obj) {
        if (obj == null)
            return false;
        if (!(obj.getClass().equals(this.getClass())))
            return false;
        Pair<?, ?> other = (Pair<?, ?>) obj;
        if (other.first != first && (other.first == null || !other.first.equals(first)))
            return false;
        if (other.second != second && (other.second == null || !other.second.equals(second)))
            return false;
        return true;
    }
    
    @Override
    public int hashCode() {
        return hash;
    }
    
    @Override
    public String toString() {
        return "<"+first+", "+second+">";
    }
    
    private int makeHash() {
        return (first == null ? 0 : first.hashCode()) + (second == null ? 0 : second.hashCode())*31;
    }

    @SuppressWarnings("unchecked")
    @Override
    public int compareTo(Pair<T1, T2> arg0) {
        if (first instanceof Comparable && second instanceof Comparable) {
            int c1 = ((Comparable<T1>)first).compareTo(arg0.first);
            if (c1 != 0)
                return c1;
            return ((Comparable<T2>)second).compareTo(arg0.second);
        }
        return hash - arg0.hash;
    }

}
