/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.datastructures;

import java.lang.reflect.Array;
import java.util.Collection;

/**
 * Some generic utility operations for arrays that do not exist in the standard
 * library.
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 * @author Tuukka Lehtonen
 */
public class Arrays {

    /**
     * Checks if an object is in an unsorted array.
     * @param <T>
     * @param array
     * @param object
     * @return <code>true</code> if the array contains the object
     */
    public static <T> boolean contains(T[] array, T object) {
        for (T o : array) {
            if(o.equals(object))
                return true;
        }
        return false;
    }

    /**
     * Returns index of an object in a array.
     * @param <T>
     * @param array
     * @param object
     * @return -1 if object is not in array
     */
    public static <T> int indexOf(T[] array, T object) {
        for (int i = 0; i < array.length; i++)
            if (array[i].equals(object))
                return i;
        return -1;
    }

    /**
     * Adds all objects in an array to a collection
     * @param <T>
     * @param collection
     * @param array
     */
    public static <T> void addAll(Collection<T> collection, T[] array) {
        for (T o : array)
            collection.add(o);
    }

    /**
     * Appends a single element to the specified array.
     * 
     * @param <T> type of the array elements
     * @param src the array to append to
     * @param t the element to append
     * @return new array containing the specified element at the end
     */
    public static <T> T[] append(T[] src, T t) {
        int len = src.length;
        @SuppressWarnings("unchecked")
        T[] result = (T[]) Array.newInstance(src.getClass().getComponentType(), len + 1);
        System.arraycopy(src, 0, result, 0, len);
        result[len] = t;
        return result;
    }

    /**
     * Appends a single element to the specified array.
     * 
     * @param <T> type of the array elements
     * @param src the array to append to
     * @param ts the elements to append to the array
     * @return new array containing the specified elements at the end
     */
    public static <T> T[] append(T[] src, @SuppressWarnings("unchecked") T... ts) {
        if (ts.length == 0)
            return src;
        int len = src.length;
        @SuppressWarnings("unchecked")
        T[] result = (T[]) Array.newInstance(src.getClass().getComponentType(), len + ts.length);
        System.arraycopy(src, 0, result, 0, len);
        System.arraycopy(ts, 0, result, len, ts.length);
        return result;
    }

    /**
     * Removes the specified index
     * 
     * @param <T>
     * @param src
     * @param index
     * @return
     */
    public static <T> T[] remove(T[] src, int index) {
        int len = src.length;
        if (index < 0)
            throw new ArrayIndexOutOfBoundsException("cannot remove negative index: " + index);
        if (index >= len)
            throw new ArrayIndexOutOfBoundsException("cannot remove array element " + index + ", array has " + len
                    + " elements");
        @SuppressWarnings("unchecked")
        T[] result = (T[]) Array.newInstance(src.getClass().getComponentType(), len - 1);
        System.arraycopy(src, 0, result, 0, index);
        System.arraycopy(src, index + 1, result, index, len - index - 1);
        return result;
    }

    /**
     * Like {@link #append(Object[], Object)} but checks that the element does
     * not already exist in the array before appending. Just returns the
     * specified array if the element already exists.
     * 
     * @param <T> type of the array elements
     * @param src the array to append to
     * @param t the element to append
     * @return new array containing the specified element at the end or
     *         <code>src</code> if the element was already contained
     */
    public static <T> T[] appendIfMissing(T[] src, T t) {
        int len = src.length;
        for (int i = 0; i < len; i++) {
            if (t.equals(src[i]))
                return src;
        }
        return append(src, t);
    }

    /**
     * Looks for the first occurrence of the specified element from the
     * specified array. If not found, the specified array is returned.
     * Equals-comparison is used to look for the element.
     * 
     * @param <T> type of the array elements
     * @param src the array to append to
     * @param t the element to remove
     * @return new array not containing the specified element or
     *         <code>src</code> if the element was not contained
     */
    public static <T> T[] remove(T[] src, T t) {
        int len = src.length;
        for (int i = 0; i < len; i++) {
            if (t.equals(src[i])) {
                return remove(src, i);
            }
        }
        return src;
    }

}
