/*******************************************************************************
 * Copyright (c) 2007- VTT Technical Research Centre of Finland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench.handler;

import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.Command;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.State;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.commands.ICommandService;
import org.eclipse.ui.handlers.HandlerUtil;
import org.simantics.Simantics;
import org.simantics.db.ChangeSetIdentifier;
import org.simantics.db.Operation;
import org.simantics.db.Session;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.NoHistoryException;
import org.simantics.db.service.ManagementSupport;
import org.simantics.db.service.UndoRedoSupport;
import org.simantics.ui.states.TrackedTextState;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * @author Kalle Kondelin
 */
public class SessionUndoHandler extends AbstractHandler {
    
    private final boolean DEBUG = false;
    private final boolean ENABLED = true;

    @Override
    public Object execute(ExecutionEvent e) throws ExecutionException {
        if (DEBUG)
            System.out.println("--\nUndo handler called.");

        IWorkbenchPart part = HandlerUtil.getActivePart(e);
        if (part == null)
            return null;
        IActionBars actionBars = WorkbenchUtils.getActionBars(part);
        final Session session = Simantics.peekSession();
        if (session == null)
            return null;
        
        ICommandService service = (ICommandService) PlatformUI.getWorkbench().getService(ICommandService.class);
        Command command = service.getCommand( TrackedTextState.COMMAND_ID );
        State state = command.getState( TrackedTextState.STATE_ID );
        
        boolean sessionUndoEnabled = true;
        Object value = state.getValue();
        if (value != null && value instanceof Boolean)
            sessionUndoEnabled = (Boolean) value;
        
        if (ENABLED) {
            if (sessionUndoEnabled) {
                try {
                    final AtomicReference<String> msg = new AtomicReference<String>();
                    IRunnableWithProgress undo = new IRunnableWithProgress() {
                        @Override
                        public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
                            try {
                                monitor.beginTask("Undo", IProgressMonitor.UNKNOWN);
                                msg.set( undo( session ) );
                            } catch (NoHistoryException e) {
                                msg.set("Nothing to undo.");
                            } catch (DatabaseException e) {
                                throw new InvocationTargetException(e);
                            } finally {
                                monitor.done();
                            }
                        }
                    };
    
                    // busyCursorWhile does not work because locking the session for undo
                    // will also lock the UI completely.
                    //PlatformUI.getWorkbench().getProgressService().busyCursorWhile(undo);
                    Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
                    new ProgressMonitorDialog(shell).run(true, false, undo);
                    actionBars.getStatusLineManager().setMessage( msg.get() );
                } catch (InvocationTargetException e1) {
                    throw new ExecutionException("Undo failed, database failure.", e1.getCause());
                } catch (InterruptedException e1) {
                    throw new ExecutionException("Undo failed, interrupted.", e1);
                }
            }
        } else {
            ErrorLogger.getDefault().log(IStatus.INFO, 0, "The undo command is disabled", null);
        }
        return null;
    }

    public static String getComment(Session session, ChangeSetIdentifier id) {
        byte[] data = id.getMetadata().get(CommentMetadata.class.getName());
        if(data == null)
            return "Undescribed operation.";
        String comment = CommentMetadata.deserialise(session, data).toString().trim();
        if(comment.isEmpty())
            return "Undescribed operation.";
        return comment;
    }
    
    private String undo(Session session) throws DatabaseException {
        UndoRedoSupport support = session.getService(UndoRedoSupport.class);
        
        List<Operation> ops = support.undoAndReturnOperations(session, 1);
        if(ops.isEmpty())
            return "Undo history is empty.";        
        
        Operation mainOperation = ops.get(0);
        
        String msg = null;
        long csId = mainOperation.getCSId();

        ManagementSupport management = session.getService(ManagementSupport.class);
        for(ChangeSetIdentifier id : management.getChangeSetIdentifiers(csId, csId))
            if(msg == null)
                msg = "Undo reverted: " + getComment(session, id);

        if (DEBUG)
            System.out.println(msg);
        return msg;
    }

}
