/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.utils;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.PlatformUI;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.utils.Container;

/**
 * @author Tuukka Lehtonen
 */
public class ResourceAdaptionUtils {

    /**
     * Tries to adapt an object into a {@link Resource}. This is successful if
     * the object already is a {@link ResourceContainer} or it is adaptable into
     * a {@link Resource} via {@link IAdaptable}.
     * 
     * @param o any object
     * @return a Resource or <code>null</code> if not adaptable to
     *         {@link Resource}
     */
    public static Resource adaptToResource(Object o) {
        if (o instanceof Container<?>) {
            Object obj = ((Container<?>) o).get();
            if (obj instanceof Resource)
                return (Resource) obj;
        } else if (o instanceof IAdaptable) {
            IAdaptable a = (IAdaptable) o;
            Resource r = (Resource) a.getAdapter(Resource.class);
            return r;
        }
        return null;
    }

    public static Resource adaptToResource(ReadGraph graph, Object o) throws DatabaseException {
    	Resource r = adaptToResource(o);
    	if(r != null) return r;
    	if(o instanceof Variable) {
    		Variable var = (Variable)o;
    		return var.getPossibleRepresents(graph);
    	} else if (o instanceof IAdaptable) {
            IAdaptable a = (IAdaptable) o;
            Variable var = (Variable) a.getAdapter(Variable.class);
            if(var != null)
            	return var.getPossibleRepresents(graph);
        }
    	return null;
    }
    
    private static Resource[] toResources(IStructuredSelection iss) {
        if (iss == null || iss.isEmpty())
            return Resource.NONE;

        ArrayList<Resource> list = new ArrayList<Resource>(iss.size());
        for (Iterator<?> iterator = iss.iterator(); iterator.hasNext();) {
            Object o = iterator.next();
            Resource r = adaptToResource(o);
            if (r != null)
                list.add(r);
        }
        return list.toArray(Resource.NONE);
    }

    /**
     * Tries to adapt any Object into a {@link Resource}. Successful if the
     * object is an IStructuredSelection of size one and its content is
     * adaptable to a {@link Resource} according to {@link #adaptToResource(Object)},
     * or the object itself is adaptable by {@link #adaptToResource(Object)}.
     * 
     * @param s any selection
     * @return a Resource, or <code>null</code> if not adaptable to Resource
     *         or selection size is not one.
     */
    public static Resource toSingleResource(Object o) {
        if (o instanceof IStructuredSelection) {
            IStructuredSelection iss = (IStructuredSelection) o;
            if (iss.size() != 1)
                return null;
            Object element = iss.getFirstElement();
            return adaptToResource(element);
        }
        return adaptToResource(o);
    }

    public static Resource toSingleResource(ReadGraph graph, Object o) throws DatabaseException {
        if (o instanceof IStructuredSelection) {
            IStructuredSelection iss = (IStructuredSelection) o;
            if (iss.size() != 1)
                return null;
            Object element = iss.getFirstElement();
            return adaptToResource(graph, element);
        }
        return adaptToResource(graph, o);
    }
    
    /**
     * Tries to adapt any Object into an array of {@link Resource}s. Only
     * successful if {@link #adaptToResource(Object)} succeeds or the object is an
     * IStructuredSelection that contains something adaptable to resources.
     * 
     * @param s any selection
     * @return every {@link Resource} that was adaptable or an array of size 0
     *         if nothing could be adapted.
     */
    public static Resource[] toResources(Object o) {
        if (o instanceof IStructuredSelection) {
            return toResources((IStructuredSelection) o);
        }
        Resource r = adaptToResource(o);
        return (r == null) ? Resource.NONE : new Resource[] { r };
    }

    ///////////////////////////////////////////////////////////////////////////

    /**
     * Tries to adapt an object into a {@link ResourceArray}. This is successful if
     * the object already is a {@link ResourceContainer} or it is adaptable into
     * a {@link ResourceArray} via {@link IAdaptable}.
     * 
     * @param o any object
     * @return a Resource or <code>null</code> if not adaptable to
     *         {@link Resource}
     */
    private static ResourceArray adaptToResourceArray(Object o) {
        if (o instanceof Container<?>) {
            Object obj = ((Container<?>) o).get();
            if (obj instanceof ResourceArray)
                return (ResourceArray) obj;
            if (obj instanceof Resource)
                return new ResourceArray((Resource) obj);
        }
        if (o instanceof Resource[]) {
            return new ResourceArray((Resource[]) o);
        }
        if (o instanceof ResourceArray) {
            return (ResourceArray) o;
        } else if (o instanceof IAdaptable) {
            IAdaptable a = (IAdaptable) o;
            ResourceArray r = (ResourceArray) a.getAdapter(ResourceArray.class);
            if (r!=null) return r;
            Resource[] r3 = (Resource[]) a.getAdapter(Resource[].class);
            if (r3!=null) return new ResourceArray(r3);
            Resource r2 = (Resource) a.getAdapter(Resource.class);
            if (r2!=null) return new ResourceArray(r2);
        }
        return null;
    }

    /**
     * Tries to adapt an object into a {@link ResourceArray}. This is successful if
     * the object already is a {@link ResourceContainer} or it is adaptable into
     * a {@link ResourceArray} via {@link IAdaptable}.
     * 
     * @param o any object
     * @return a Resource or <code>null</code> if not adaptable to
     *         {@link ResourceArray[]}
     */
    private static ResourceArray[] adaptToResourceArrays(Object o) {
        if (o instanceof ResourceArray[]) {        	
            return (ResourceArray[]) o;
        } else if (o instanceof IAdaptable) {
            IAdaptable a = (IAdaptable) o;
            ResourceArray[] ras = (ResourceArray[]) a.getAdapter(ResourceArray[].class);
            if (ras!=null) return ras;
        }
        ResourceArray ra = adaptToResourceArray(o);
        return ra != null ? new ResourceArray[] { ra } : null;
    }

    private static ResourceArray[] toResourceArrays(IStructuredSelection iss) {
        if (iss == null || iss.isEmpty())
            return ResourceArray.NONE;

        ArrayList<ResourceArray> list = new ArrayList<ResourceArray>(iss.size());
        for (Iterator<?> iterator = iss.iterator(); iterator.hasNext();) {
            Object o = iterator.next();
            ResourceArray[] r = adaptToResourceArrays(o);
            if (r != null)
                for (ResourceArray ra : r)
                    list.add(ra);
        }
        return list.toArray(ResourceArray.NONE);
    }

    /**
     * Tries to adapt any Object into a {@link Resource}. Successful if the
     * object is an IStructuredSelection of size one and its content is
     * adaptable to a {@link Resource} according to {@link #adaptToResource(Object)},
     * or the object itself is adaptable by {@link #adaptToResource(Object)}.
     * 
     * @param s any selection
     * @return a Resource, or <code>null</code> if not adaptable to Resource
     *         or selection size is not one.
     */
    public static ResourceArray toSingleResourceArray(Object o) {
        if (o instanceof IStructuredSelection) {
            IStructuredSelection iss = (IStructuredSelection) o;
            if (iss.size() != 1)
                return null;
            Object element = iss.getFirstElement();
            return adaptToResourceArray(element);
        }
        return adaptToResourceArray(o);
    }

    /**
     * Tries to adapt any Object into an array of {@link Resource}s. Only
     * successful if the input object is directly adaptable to ResourceArray[],
     * if {@link #adaptToResourceArray(Object)} succeeds or the object is an
     * IStructuredSelection containing resources in some form.
     * 
     * @param s any selection
     * @return every {@link Resource} that was adaptable or an array of size 0
     *         if nothing could be adapted.
     */
    public static ResourceArray[] toResourceArrays(Object o) {
        if (o instanceof List<?>) {
            List<?> list = (List<?>)o;
            ArrayList<ResourceArray> out = new ArrayList<ResourceArray>(list.size());
            for (Object i : list) {
                ResourceArray[] r = adaptToResourceArrays(i);
                if (r != null)
                    for (ResourceArray ra : r)
                        out.add(ra);
            }
            return out.toArray(ResourceArray.NONE);
        }
        if (o instanceof IStructuredSelection) {
            return toResourceArrays((IStructuredSelection) o);
        }
        ResourceArray r = adaptToResourceArray(o);
        return (r == null) ? ResourceArray.NONE : new ResourceArray[] { r };
    }

    public static Resource toSingleWorkbenchResource() {
        return toSingleResource(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getSelectionService().getSelection());
    }

}
