/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.trend.configuration;

import java.util.ArrayList;
import java.util.List;

import org.simantics.databoard.Datatypes;
import org.simantics.databoard.type.RecordType;
import org.simantics.history.util.subscription.SamplingFormat;

public class TrendSamplingFormats {

	// Template Sampling formats	
	public final static SamplingFormat analog_all, analog_100ms, analog_1s, analog_10s, analog_60s, analog_minmax; 
	public final static SamplingFormat binary_all, binary_100ms, binary_1s, binary_10s, binary_60s;
	public final static SamplingFormat time;
	
	public final static String all="all", _100ms="100ms", _1s="1s", _10s="10s", _60s="60s", minmax="minmax"; 
	
	/**
	 * Create a list of sampling formats for trend's needs 
	 * 
	 * @param interval
	 * @param deadband
	 * @return
	 */
	public static List<SamplingFormat> createAnalogSamplingFormats( double interval, double deadband, String unit )
	{
		ArrayList<SamplingFormat> result = new ArrayList<SamplingFormat>( 6 );
		
		result.add( analog_minmax );
		
		if (interval==0. && deadband==0.) {
			result.add( analog_all.clone().setUnit(unit) );
			result.add( analog_100ms.clone().setUnit(unit) );
			result.add( analog_1s.clone().setUnit(unit) );
			result.add( analog_10s.clone().setUnit(unit) );
			result.add( analog_60s.clone().setUnit(unit) );
			return result;
		} 

		// Format for ]0,1 .. 1[
		if (interval < 0.1) {
			result.add( analog_all.clone( interval, deadband ).setUnit(unit) );
			result.add( analog_100ms.clone( 0.1, deadband ).setUnit(unit) );
			result.add( analog_1s.clone( 1, deadband).setUnit(unit) );
			result.add( analog_10s.clone( 10, deadband).setUnit(unit) );
			result.add( analog_60s.clone( 60, deadband).setUnit(unit) );
			return result;
		}
		
		// Format for ]0 .. 1[
		if (interval < 1.) {
			result.add( analog_100ms.clone( interval, deadband ).setUnit(unit) );
			result.add( analog_1s.clone( 1, deadband).setUnit(unit) );
			result.add( analog_10s.clone( 10, deadband).setUnit(unit) );
			result.add( analog_60s.clone( 60, deadband).setUnit(unit) );
			return result;
		}

		// Format for [1 .. 10[
		if (interval < 10.) {
			result.add( analog_1s.clone( interval, deadband).setUnit(unit) );
			result.add( analog_10s.clone( 10, deadband).setUnit(unit) );
			result.add( analog_60s.clone( 60, deadband).setUnit(unit) );
			return result;
		}

		// Format for [10 .. 60[
		if (interval < 60.) {
			result.add( analog_10s.clone( interval, deadband).setUnit(unit) );
			result.add( analog_60s.clone( 60, deadband).setUnit(unit) );
			return result;
		}

		// Format for >60s
		result.add( analog_60s.clone( interval, deadband).setUnit(unit) );
		return result;
	}

	/**
	 * Create a list of sampling formats for trend's needs 
	 * 
	 * @param interval
	 * @return
	 */
	public static List<SamplingFormat> createBinarySamplingFormats( double interval )
	{
		ArrayList<SamplingFormat> result = new ArrayList<SamplingFormat>( 5 );
		
		if (interval==0.) {
			result.add( binary_all );
			result.add( binary_100ms );
			result.add( binary_1s );
			result.add( binary_10s );
			result.add( binary_60s );
			return result;
		} 

		// Format for ]0,1 .. 1[
		if (interval < 1.) {
			result.add( binary_all.clone( interval, Double.NaN ) );
			result.add( binary_100ms );
			result.add( binary_1s );
			result.add( binary_10s );
			result.add( binary_60s );
			return result;
		}
		
		// Format for ]0 .. 1[
		if (interval < 1.) {
			result.add( binary_100ms.clone( interval, Double.NaN ) );
			result.add( binary_1s );
			result.add( binary_10s );
			result.add( binary_60s );
			return result;
		}

		// Format for [1 .. 10[
		if (interval < 10.) {
			result.add( binary_1s.clone( interval, Double.NaN) );
			result.add( binary_10s );
			result.add( binary_60s );
			return result;
		}

		// Format for [10 .. 60[
		if (interval < 60.) {
			result.add( binary_10s.clone( interval, Double.NaN) );
			result.add( binary_60s );
			return result;
		}

		// Format for >60s
		result.add( binary_60s.clone( interval, Double.NaN) );
		return result;
	}
	
	static {
		time = new SamplingFormat();
		RecordType format = (RecordType) (time.format = new RecordType());
		format.addComponent("time", Datatypes.DOUBLE);
		format.addComponent("value", Datatypes.DOUBLE);
		format.addComponent("quality", Datatypes.BYTE);
		time.interval = Double.NaN;
		time.deadband = Double.NaN;

		analog_all = new SamplingFormat();
		analog_all.formatId = all;
		format = (RecordType) (analog_all.format = new RecordType());
		format.addComponent("time", Datatypes.DOUBLE);
		format.addComponent("endTime", Datatypes.DOUBLE);
		format.addComponent("value", Datatypes.DOUBLE);
		format.addComponent("quality", Datatypes.BYTE);
		analog_all.interval = Double.NaN;
		analog_all.deadband = Double.NaN;
		
		analog_1s = new SamplingFormat();
		analog_1s.formatId = _1s;
		analog_1s.format = new RecordType();
		format = (RecordType) (analog_1s.format = new RecordType());
		format.addComponent("time", Datatypes.DOUBLE);
		format.addComponent("endTime", Datatypes.DOUBLE);		
		format.addComponent("value", Datatypes.DOUBLE);
		format.addComponent("min", Datatypes.DOUBLE);
		format.addComponent("max", Datatypes.DOUBLE);
		format.addComponent("avg", Datatypes.DOUBLE);		
		//format.addComponent("median", Datatypes.DOUBLE);		
		format.addComponent("quality", Datatypes.BYTE);
		analog_1s.interval = 1.0;
		analog_1s.deadband = Double.NaN;		
		
		analog_100ms = analog_1s.cloneTo(_100ms, 0.1, Double.NaN);		
		analog_10s = analog_1s.cloneTo(_10s, 10., Double.NaN);		
		analog_60s = analog_10s.cloneTo(_60s, 60., Double.NaN);

		analog_minmax = new SamplingFormat();
		analog_minmax.formatId = minmax;
		format = (RecordType) (analog_minmax.format = new RecordType());
		format.addComponent("time", Datatypes.DOUBLE);
		format.addComponent("endTime", Datatypes.DOUBLE);
		format.addComponent("value", Datatypes.DOUBLE);
		format.addComponent("min", Datatypes.DOUBLE);
		format.addComponent("max", Datatypes.DOUBLE);
		analog_minmax.interval = Double.MAX_VALUE;
		analog_minmax.deadband = Double.MAX_VALUE;
		
		binary_all = new SamplingFormat();
		binary_all.formatId = all;
		format = (RecordType) (binary_all.format = new RecordType());
		format.addComponent("time", Datatypes.DOUBLE);
		format.addComponent("endTime", Datatypes.DOUBLE);
		format.addComponent("value", Datatypes.BYTE);
		format.addComponent("max", Datatypes.BYTE);
		format.addComponent("quality", Datatypes.BYTE);
		binary_all.interval = Double.NaN;
		binary_all.deadband = Double.NaN;
		
		binary_1s = new SamplingFormat();
		binary_1s.formatId = _1s;
		format = (RecordType) (binary_1s.format = new RecordType());
		format.addComponent("time", Datatypes.DOUBLE);
		format.addComponent("endTime", Datatypes.DOUBLE);		
		format.addComponent("value", Datatypes.BYTE);
		format.addComponent("max", Datatypes.BYTE);
		//format.addComponent("median", Datatypes.BYTE);		
		format.addComponent("quality", Datatypes.BYTE);
		binary_1s.interval = 1.0;
		binary_1s.deadband = Double.NaN;		
		
		binary_100ms = binary_1s.cloneTo(_100ms, 0.1, Double.NaN);		
		binary_10s = binary_1s.cloneTo(_10s, 10., Double.NaN);		
		binary_60s = binary_1s.cloneTo(_60s, 60., Double.NaN);
	}
	
	
}
