package org.simantics.team.ui;

import java.io.File;
import java.io.IOException;
import java.util.Deque;
import java.util.LinkedList;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.preference.IPersistentPreferenceStore;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IImportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.team.Activator;
import org.simantics.team.Utils;
import org.simantics.utils.ui.ErrorLogger;

public class StageInitWizard extends Wizard implements IImportWizard {
    static public class Data {
        public Boolean ok = false;
        public Boolean requireExisting = false;
        public String comment; // Initial comment.
        Deque<String> recentLocations = new LinkedList<String>();;
        final File stagingFolder;
        public File teamFolder;
        public Data(File stagingFolder, File teamFolder) {
            this.stagingFolder = stagingFolder;
            this.teamFolder = teamFolder;
        }
    }
    public static boolean openInitWizard(final Data data)
    throws DatabaseException {
        final Display display = PlatformUI.getWorkbench().getDisplay();
        display.syncExec(new Runnable() {
            @Override
            public void run() {
                StageInitWizard stageInitWizard = new StageInitWizard(data);
                WizardDialog dialog = new WizardDialog(display.getActiveShell(), stageInitWizard);
                if (Dialog.OK != dialog.open())
                    data.ok = false;
                else
                    data.ok = true;
            }});
        return data.ok;
    }
    private final Data data;
    private static final int MAX_RECENT_IMPORT_PATHS = 10;
    private boolean readPreferences(IStructuredSelection selection) {
        IPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);
        try {
            data.teamFolder = Utils.getTeamFolder();
            if (null == data.teamFolder)
                data.teamFolder = new File("Default Team Folder");
        } catch (DatabaseException e) {
            Logger.defaultLogError("Failed to get team folder.", e);
            data.teamFolder = new File("Default Team Folder");
        }
        String recentPathsPref = store.getString(Preferences.RECENT_TEAM_FOLDERS);
        data.recentLocations = Preferences.decodePaths(recentPathsPref);
        return true;
    }
    private void writePreferences() throws IOException {
        IPersistentPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);
        store.putValue(Preferences.CURRENT_TEAM_FOLDER, data.teamFolder.getAbsolutePath());
        store.putValue(Preferences.RECENT_TEAM_FOLDERS, Preferences.encodePaths(data.recentLocations));
        if (store.needsSaving())
            store.save();
    }
    public StageInitWizard(Data data) {
        this.data = data;
        setWindowTitle("Staging Init");
        setNeedsProgressMonitor(true);
    }
    @Override
    public void init(IWorkbench workbench, IStructuredSelection selection) {
        readPreferences(selection);
    }
    @Override
    public void addPages() {
        super.addPages();
        addPage(new StageInitPage(data));
    }
    @Override
    public boolean performFinish() {
        try {
            data.recentLocations.addFirst(data.teamFolder.getAbsolutePath());
            Preferences.removeDuplicates(data.recentLocations);
            if (data.recentLocations.size() > MAX_RECENT_IMPORT_PATHS)
                data.recentLocations.pollLast();
            writePreferences();
        } catch (IOException e) {
            ErrorLogger.defaultLogError("Failed to write preferences", e);
        }
        return true;
    }
}
