/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.spreadsheet.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

import org.apache.poi.ss.usermodel.DataFormatter;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.mutable.MutableVariant;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.datatypes.utils.BTree;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.PossibleChild;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.AssumptionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.VariableException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.layer0.variable.ProxyVariables;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.VariableSpaceManipulator.PropertyCreationData;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.request.Write;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.document.server.io.IColor;
import org.simantics.document.server.io.IFont;
import org.simantics.document.server.io.RGBColor;
import org.simantics.document.server.io.SimpleFont;
import org.simantics.layer0.Layer0;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.scl.runtime.tuple.Tuple;
import org.simantics.spreadsheet.ClientModel;
import org.simantics.spreadsheet.OperationMode;
import org.simantics.spreadsheet.Range;
import org.simantics.spreadsheet.Spreadsheets;
import org.simantics.spreadsheet.Transaction;
import org.simantics.spreadsheet.common.TableCell;
import org.simantics.spreadsheet.common.cell.StringCellParser;
import org.simantics.spreadsheet.resource.SpreadsheetResource;
import org.simantics.utils.datastructures.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SpreadsheetUtils {

	// This is a variable under the active run when in Operation Mode 
	public static String SPREADSHEET_TRANSACTION_CONTEXT_VARIABLE = "variable";
	public static String SPREADSHEET_TRANSACTION_CONTEXT_GRAPH = "graph";

	private static final Logger LOGGER = LoggerFactory.getLogger(SpreadsheetUtils.class);

	public static Pair<String, Collection<PropertyCreationData>> parse(String text, StringCellParser[] parsers) {

		try {

			for(StringCellParser parser : parsers) { 
				Collection<PropertyCreationData> parsed = parser.parse(text);
				if(parsed != null) return Pair.make(parser.getType(), parsed);
			}

		} catch (Throwable t) {
			t.printStackTrace();
		}

		return null;

	}

	public static boolean isImmutable(Object object) {
		return !(object instanceof Resource) && !(object instanceof Variable); 
	}

	public static String getLabel(ReadGraph graph, Object object) throws DatabaseException {

		if(object == null) {
			return "no data";
		}

		if(object instanceof Resource) {
			return NameUtils.getSafeName(graph, (Resource)object);
		} else if (object instanceof Variable) {
			try {
				Object value = ((Variable)object).getValue(graph);
				return value.toString();
				//return toString(value);
			} catch (VariableException e) {
				Object value = ((Variable)object).getPropertyValue(graph, "Label"); 
				return value.toString();
			}
		} else if (object instanceof double[]) {
			return object.toString();
			//			return toString(object); 
		} else {
			return object.toString();
		}

	}

	private static String toString(Object object) {
		if(object instanceof double[]) {
			try {
				return Bindings.DOUBLE_ARRAY.toString(object);
			} catch (BindingException e) {
				return object.toString();
			}
		} else {
			return object.toString();
		}
	}

	public static String getContent(ReadGraph graph, Object object) throws DatabaseException {

		if(object == null) {
			return null;
		}

		if(object instanceof Resource) {
			SerialisationSupport support = graph.getService(SerialisationSupport.class);
			return support.getResourceSerializer().createRandomAccessId((Resource)object);
		} else if (object instanceof Variable) {
			return ((Variable)object).getURI(graph);
		} else {
			return "";
		}

	}

	public static void main(String[] args) {
		for(int i=0;i<16384;i++) {
			String name = Spreadsheets.columnName(i);
			Range r = Spreadsheets.decodeCellAbsolute(name + "1");
			System.err.println(i + " " + name + " " + r);
		}
	}

	public static String getLabel(ClientModel model, int row, int column) {
		try {
			String location = Spreadsheets.cellName(row, column);
			String label = model.getPropertyAt(location, ClientModel.LABEL);
			if(label != null) return label;
			Variant content = SpreadsheetUtils.getSafeClientVariant(model, location, ClientModel.CONTENT);
			if(content != null) return SpreadsheetUtils.getContentString(content);
			else return null;
		} catch (Throwable e) {
			e.printStackTrace();
			return null;
		}
	}

	public static String getContentString(Variant content) {
		return content.getValue().toString();
	}

	public static boolean isInBounds(String base, String location, int wBounds, int hBounds) {
		Range baseRange = Spreadsheets.decodeCellAbsolute(base);
		Range locationRange = Spreadsheets.decodeCellAbsolute(location);
		if(locationRange.startColumn < baseRange.startColumn) return false;
		if(locationRange.startRow < baseRange.startRow) return false;
		int wb = wBounds == -1 ? (Integer.MAX_VALUE / 3) : wBounds;
		int hb = hBounds == -1 ? (Integer.MAX_VALUE / 3) : hBounds;
		if(locationRange.startColumn > (baseRange.startColumn+wb-1)) return false;
		if(locationRange.startRow > (baseRange.startRow+hb-1)) return false;
		return true;
	}

	public static void schedule(Transaction<?> transaction, Write write) {

		if(transaction == null) {

			TransactionImpl impl = (TransactionImpl)startTransaction(OperationMode.OPERATION);
			impl.add(write);
			impl.commit();

		} else {

			TransactionImpl impl = (TransactionImpl)transaction;
			impl.add(write);

		}

	}
	
	public static Transaction<Write> startTransaction() {
		return startTransaction(OperationMode.EDIT_MODE);
	}

	public static Transaction<Write> startTransaction(OperationMode mode) {
		return new TransactionImpl(mode);
	}	

	static class TransactionImpl implements Transaction<Write> {

		private ArrayList<Write> writes = new ArrayList<>();
		private final OperationMode mode;
		private Map<String,Object> context = new HashMap<>();
		private List<Object> needSync;
		private Set<Consumer<Transaction<Write>>> commitListeners = new HashSet<>(); 

		public TransactionImpl(OperationMode mode) {
			this.mode = mode;
		}

		public void commit(boolean sync) throws Exception {

			RequestProcessor processor = getContext(SPREADSHEET_TRANSACTION_CONTEXT_GRAPH);
			if(processor == null)
				processor = Simantics.getSession();

			WriteRequest req = new WriteRequest() {

				@Override
				public void perform(WriteGraph graph) throws DatabaseException {
					graph.markUndoPoint();
					for(int i=0;i<writes.size();i++) {
						Write write = writes.get(i);
						try {
							write.perform(graph);
						} catch (DatabaseException e) {
							LOGGER.error("Error while committing Spreadsheet transaction", e);
						}
						// This can schedule more writes
						//graph.syncRequest(write);
					}
					writes.clear();
				}
			};

			if(sync) {
				processor.sync(req);
				// Make sure that all ExternalReads have been finished
				processor.sync(new WriteRequest() {

					@Override
					public void perform(WriteGraph graph) throws DatabaseException {
					}
					
				});
				for(Consumer<Transaction<Write>> listener : commitListeners)
					listener.accept(this);;
			}
			else
				processor.async(req, new Procedure<Object>() {

					void fire() {
						Simantics.async(() -> {
							for(Consumer<Transaction<Write>> listener : commitListeners)
								listener.accept(TransactionImpl.this);
						});
					}
					
					@Override
					public void exception(Throwable t) {
						LOGGER.error("Error while committing Spreadsheet transaction", t);
						fire();
					}

					@Override
					public void execute(Object result) {
						fire();
					}
					
				});
			
		}
		
		@Override
		public void commit() {
			try {
				commit(false);
			} catch (Exception e) {
				LOGGER.error("Error while committing Spreadsheet transaction", e);
			}
		}

		@Override
		public void add(Write write) {
			writes.add(write);
		}

		@Override
		public boolean isOperationMode() {
			return mode.equals(OperationMode.OPERATION);
		}

		@Override
		public void setContext(String key, Object context) {
			this.context.put(key, context);
		}

		@Override
		public <T> T getContext(String key) {
			return (T)context.get(key);
		}

		@Override
		public void needSynchronization(Object object) {
			if (needSync == null)
				needSync = new ArrayList<>();
			needSync.add(object);
		}

		@Override
		public List<Object> needSynchronization() {
			return needSync;
		}
		
		@Override
		public void addCommitListener(Consumer<Transaction<Write>> listener) {
			commitListeners.add(listener);
		}

	}

	public static MutableVariant createVariant() {
		return new MutableVariant();
	}

	public static Variant getSafeClientVariant(ClientModel clientModel, String location, String property) {
		try {
			return clientModel.getPossiblePropertyAt(location, property);
		} catch (Throwable t) {
			LOGGER.error("Error while obtaining Spreadsheet value", t);
			return Variant.ofInstance(t.getMessage());
		}
	}

    public static Resource createSheet(WriteGraph graph, Resource book, String name) throws DatabaseException {
    	
    	return createSheet(graph, book, name, new String[] {}, new int[] {});
    	
    }

	public static Resource createSheet(WriteGraph graph, Resource book, String name, String[] colNames, int[] colWidths) throws DatabaseException {

        Layer0 L0 = Layer0.getInstance(graph);
        SpreadsheetResource sr = SpreadsheetResource.getInstance(graph);

        Resource result = graph.newResource();
        graph.claim(result, L0.InstanceOf, null, sr.Spreadsheet);

        if(name == null) {
            name = NameUtils.findFreshEscapedName(graph, "Sheet", book, L0.ConsistsOf);
        }
        graph.claimLiteral(result, L0.HasName, L0.NameOf, L0.String, name, Bindings.STRING);
        graph.claim(book, L0.ConsistsOf, L0.PartOf, result);

//            Resource newCell = graph.newResource();
//            graph.claim(newCell, L0.InstanceOf, null, sr.Lines);
//            graph.claimLiteral(newCell, L0.HasName, L0.NameOf, L0.String, "Lines", Bindings.STRING);
//            graph.claim(result, L0.ConsistsOf, L0.PartOf, newCell);
//            BTree bt = new BTree(graph, SpreadsheetUtils.SPREADSHEET_BTREE_SIZE, SR.Lines, SR.LineNode, L0.PartOf, true);
        
        BTree bt = new BTree(graph, Spreadsheets.SPREADSHEET_BTREE_SIZE, sr.Lines, sr.LineNode, L0.PartOf, false);
//        BTree bt = BTreeUtils.create(graph, sr.Lines, sr.LineNode, L0.PartOf, SpreadsheetUtils.SPREADSHEET_BTREE_SIZE, false);
        Resource lines = bt.rootOfBTree();
        
        graph.claimLiteral(lines, L0.HasName, L0.NameOf, L0.String, "Lines", Bindings.STRING);
        graph.claim(result, L0.ConsistsOf, L0.PartOf, lines);
        
        {
            Resource newCell = graph.newResource();
            graph.claim(newCell, L0.InstanceOf, null, sr.Dimensions);
            graph.claimLiteral(newCell, L0.HasName, L0.NameOf, L0.String, "Dimensions", Bindings.STRING);
            graph.addLiteral(newCell, sr.Dimensions_fitColumns, sr.Dimensions_fitColumns_Inverse, L0.Boolean, false, Bindings.BOOLEAN);
            graph.addLiteral(newCell, sr.Dimensions_fitRows, sr.Dimensions_fitRows_Inverse, L0.Boolean, false, Bindings.BOOLEAN);
            graph.addLiteral(newCell, sr.Dimensions_columnCount, sr.Dimensions_columnCount_Inverse, L0.Integer, 128, Bindings.INTEGER);
            graph.addLiteral(newCell, sr.Dimensions_rowCount, sr.Dimensions_rowCount_Inverse, L0.Integer, 256, Bindings.INTEGER);
            graph.claim(result, L0.ConsistsOf, L0.PartOf, newCell);
        }

        {
            Resource newCell = graph.newResource();
            graph.claim(newCell, L0.InstanceOf, null, sr.Headers);
            graph.claimLiteral(newCell, L0.HasName, L0.NameOf, L0.String, "Headers", Bindings.STRING);
            graph.addLiteral(newCell, sr.Headers_columnLabels, sr.Headers_columnLabels_Inverse, L0.StringArray, colNames, Bindings.STRING_ARRAY);
            graph.addLiteral(newCell, sr.Headers_columnWidths, sr.Headers_columnWidths_Inverse, L0.IntegerArray, colWidths, Bindings.INT_ARRAY);
            graph.claim(result, L0.ConsistsOf, L0.PartOf, newCell);
        }

        return result;

    }
    

    public static Variable getBookVariable(ReadGraph graph, Resource book) throws DatabaseException {
    	Variable variable = Variables.getVariable(graph, book);
    	return ProxyVariables.makeProxyVariable(graph, variable, variable);
    }

    public static Variable sheetRun(ReadGraph graph, Resource book, Variable context) throws DatabaseException {
    	Variable root = Variables.getVariable(graph, book);
    	return ProxyVariables.makeProxyVariable(graph, root, context);
    }

    private static TableCell constructCell(int row, int column, Object data) {
		TableCell cell = new TableCell();
		cell.row = row;
		cell.column = column;
		cell.text = data.toString();
		return cell;
    }
    
    public static List<TableCell> queryCells(Object data) {
    	ArrayList<TableCell> result = new ArrayList<TableCell>();
    	if(data instanceof List) {
    		List<?> list = (List<?>)data;
    		int row = 0;
    		for(Object o : list) {
    			if(o instanceof Tuple) {
    				Tuple t = (Tuple)o;
    				for(int i=0;i<t.length();i++) {
    					result.add(constructCell(row, i, t.get(i)));
    				}
    			} else if (o instanceof List) {
    				List<?> rowList = (List<?>)o;
    				int index = 0;
    				for(Object obj : rowList) {
    					result.add(constructCell(row, index++, obj));
    				}
    			} else {
					result.add(constructCell(row, 0, o));
    			}
    			row++;
    		}
    	}
    	return result;
    }
    
    public static List<TableCell> organizeCells(int columns, List<String> headers_, List<TableCell> cells) throws DatabaseException {
    	
    	ArrayList<TableCell> result = new ArrayList<TableCell>();
    	
    	int w = 0; // name + fields 
    	int h = 0; // number or rows excluding headers
    	
    	if(columns < 2) throw new AssumptionException("organizeCells: number of columns needs to be greater than 1");
    	
    	for(TableCell cell : cells) {
    		if((cell.column+1)>w) w = cell.column+1;
    		if((cell.row)>h) h = cell.row;
    	}
    	
    	int fields = w - 1;
    	
    	if(columns > (fields + 1)) columns = fields + 1;//throw new DatabaseException("organizeCells: number of block columns cannot be greater than the amount of columns in data");
    	
    	int fieldsPerRow = columns - 1;
    	
    	int blocks = fields / fieldsPerRow;
    	if(fields%fieldsPerRow > 0) blocks++;

    	TableCell[] names = new TableCell[h];
    	TableCell[] headers = new TableCell[w];
    	
    	for(TableCell cell : cells) {
    		
    		if(cell.row == 0) {
    			headers[cell.column] = cell;
    		} else if(cell.column == 0) {
    			names[cell.row-1] = cell;
    		} else {
        		TableCell copy = new TableCell(cell);
        		int block = (copy.column-1) / fieldsPerRow;
        		copy.row = block*(h+1) + copy.row;
       			copy.column = 1 + (copy.column-1) % fieldsPerRow;
        		result.add(copy);
    		}
    		
    	}
    	
		for(int j=0;j<blocks;j++) {

			int rowBase = j*(h+1);
			
			for(int i=0;i<h;i++) {
        		TableCell copy = new TableCell(names[i]);
    			copy.row = rowBase + copy.row;
        		result.add(copy);
    		}
			
			TableCell legend = new TableCell(headers[0]);
			legend.row = rowBase;
			result.add(legend);

			for(int i=1;i<columns;i++) {

				int index = (j*fieldsPerRow) + i;
				if(index >= w) continue;
				
				TableCell header = new TableCell(headers[index]);
				header.row = rowBase;
				header.column = i;
				result.add(header);
				
    		}
			
    	}
    	
    	return result;
    	
    }
    
    public static List<TableCell> modifyCells1(List<TableCell> cells, Function1<TableCell, TableCell> fn) {
    	ArrayList<TableCell> result = new ArrayList<TableCell>();
    	for(TableCell cell : cells)
    		result.add(fn.apply(cell));
    	return result;
    }

    public static List<TableCell> modifyCells(List<TableCell> cells, List<Function1<TableCell, TableCell>> fns) {
    	ArrayList<TableCell> result = new ArrayList<TableCell>();
    	for(TableCell cell : cells) {
    		for(Function1<TableCell,TableCell> fn : fns)
    			cell = fn.apply(cell);
    		result.add(cell);
    	}
    	return result;
    }

    public static TableCell applyFont(IFont font, Function1<TableCell,Boolean> filter, TableCell cell) {
    	if(!filter.apply(cell)) return cell;
    	TableCell result = new TableCell(cell);
    	result.font = font;
    	return result;
    }

    public static TableCell applyAlign(int align, Function1<TableCell,Boolean> filter, TableCell cell) {
    	if(!filter.apply(cell)) return cell;
    	TableCell result = new TableCell(cell);
    	result.align = align;
    	return result;
    }

    public static TableCell applyForeground(IColor color, Function1<TableCell,Boolean> filter, TableCell cell) {
    	if(!filter.apply(cell)) return cell;
    	TableCell result = new TableCell(cell);
    	result.foreground = color;
    	return result;
    }

    public static TableCell applyBackground(IColor color,Function1<TableCell,Boolean> filter,  TableCell cell) {
    	if(!filter.apply(cell)) return cell;
    	TableCell result = new TableCell(cell);
    	result.background = color;
    	return result;
    }

    public static IFont simpleFont(String family, String style, int height) {
    	return new SimpleFont(family, style, height);
    }

    public static IColor rgbColor(int r, int g, int b) {
    	return new RGBColor(r, g, b);
    }

    public static boolean selectRow(int row, TableCell cell) {
    	return cell.row == row;
    }

    public static boolean selectColumn(int column, TableCell cell) {
    	return cell.column == column;
    }

    public static void setSCLLine(WriteGraph graph, Resource spreadsheet, int row, String expression) throws DatabaseException {

    	Layer0 L0 = Layer0.getInstance(graph);
    	Resource lines = graph.syncRequest(new PossibleChild(spreadsheet, "Lines"));
    	BTree bt = new BTree(graph, lines);
    	SpreadsheetResource SR = SpreadsheetResource.getInstance(graph);
    	
    	Resource line = graph.newResource();
    	graph.claim(line, L0.InstanceOf, SR.Line);
    	graph.addLiteral(line, L0.HasName, L0.NameOf, "" + row, Bindings.STRING);
    	Layer0Utils.setExpression(graph, line, SR.Line_content, null, "[spreadsheetCell ]", L0.SCLValue);
    	bt.insertBTree(graph, Variant.ofInstance(row), line);
    	
    }

    public static String getFormattedLabel(ClientModel model, int row, int column, int formatIndex, String formatString) {
        if (formatString == null)
            return getLabel(model, row, column); 
        try {
            String location = Spreadsheets.cellName(row, column);
            Variant content = SpreadsheetUtils.getSafeClientVariant(model, location, ClientModel.CONTENT);
            if(content != null) {
                
                String contentString = SpreadsheetUtils.getContentString(content);
                if(contentString.equals("~CIRCULAR~REF~"))
                	return "0";
                
                double value = Double.valueOf(contentString);
                if (Double.isNaN(value))
                	return getLabel(model, row, column);
                
                DataFormatter formatter = new DataFormatter();
                return formatter.formatRawCellContents(value, formatIndex, formatString);
            }
            return null;
        } catch (NumberFormatException e) {
            return getLabel(model, row, column);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    /*
     *  Please use Spreadsheets.cellName instead
     */
    @Deprecated
    public static String cellName(int row, int column) {
        return Spreadsheets.cellName(row, column);
    }

    /*
     *  Please use Spreadsheets.columnName instead
     */
    @Deprecated
    public static String columnName(int column) {
        return Spreadsheets.columnName(column);
    }

    /*
     *  Please use Spreadsheets.decodeCellAbsolute instead
     */
    @Deprecated
    public static Range decodeCellAbsolute(String identifier) {
        return Spreadsheets.decodeCellAbsolute(identifier);
    }
    
    /*
     *  Please use Spreadsheets.decodePossibleCellAbsolute instead
     */
    @Deprecated
    public static Range decodePossibleCellAbsolute(String identifier) {
        return Spreadsheets.decodePossibleCellAbsolute(identifier);
    }

    /*
     *  Please use Spreadsheets.decodeRange instead
     */
    @Deprecated
    public static Range decodeRange(String rangeOrCell) {
        return Spreadsheets.decodeRange(rangeOrCell);
    }

    /*
     *  Please use Spreadsheets.decodeRanges instead
     */
    @Deprecated
    public static List<Range> decodeRanges(String ranges) {
        return Spreadsheets.decodeRanges(ranges);
    }
    
    /*
     *  Please use Spreadsheets.startColumn instead
     */
    @Deprecated
    public static int startColumn(List<Range> ranges) {
        return Spreadsheets.startColumn(ranges);
    }

    /*
     *  Please use Spreadsheets.startRow instead
     */
    @Deprecated
    public static int startRow(List<Range> ranges) {
        return Spreadsheets.startRow(ranges);
    }
    
    /*
     *  Please use Spreadsheets.offset instead
     */
    @Deprecated
    public static String offset(String location, int rowOffset, int columnOffset) {
        return Spreadsheets.offset(location, rowOffset, columnOffset);
    }

    
}
