/*******************************************************************************
 * Copyright (c) 2015 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.internal.startup;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.dynamichelpers.ExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IExtensionChangeHandler;
import org.eclipse.core.runtime.dynamichelpers.IExtensionTracker;
import org.eclipse.core.runtime.dynamichelpers.IFilter;

/**
 * Registry implementation for org.simantics.startup extensions.
 * 
 * @author Tuukka Lehtonen
 */
public class StartupRegistry implements IExtensionChangeHandler {

    private final static String    NAMESPACE  = "org.simantics";
    private final static String    EP_NAME    = "startup";
    private final static String    STARTUP    = "startup";

    private final ExtensionTracker tracker;

    private StartupExtension[]     extensions = {};

    public StartupRegistry() {
        tracker = new ExtensionTracker();

        // Cache defined actions
        IExtensionPoint expt = Platform.getExtensionRegistry().getExtensionPoint(NAMESPACE, EP_NAME);
        loadExtensions(expt.getConfigurationElements());

        // Start tracking for new and removed extensions
        IFilter filter = ExtensionTracker.createExtensionPointFilter(expt);
        tracker.registerHandler(this, filter);
    }

    private void loadExtensions(IConfigurationElement[] configurationElements) {
        Set<StartupExtension> newExtensions = new HashSet<StartupExtension>(Arrays.asList(extensions));

        // These are all "startup" elements with required attributes
        //  - class
        for (IConfigurationElement el : configurationElements) {
            if (STARTUP.equals(el.getName())) {
                String clazz = el.getAttribute(StartupExtension.CLASS);
                if (clazz == null) {
                    // No class defined.
                    continue;
                }

                StartupExtension ext = new StartupExtension(el);

                // Start tracking the new extension object, its removal will be notified of
                // with removeExtension(extension, Object[]).
                tracker.registerObject(el.getDeclaringExtension(), ext, IExtensionTracker.REF_STRONG);

                newExtensions.add(ext);
            }
        }

        // Atomic assignment
        this.extensions = newExtensions.toArray(new StartupExtension[newExtensions.size()]);
    }

    @Override
    public void addExtension(IExtensionTracker tracker, IExtension extension) {
        loadExtensions(extension.getConfigurationElements());
    }

    @Override
    public void removeExtension(IExtension extension, Object[] objects) {
        Set<StartupExtension> newExtensions = new HashSet<StartupExtension>(Arrays.asList(extensions));

        for (Object o : objects) {
            tracker.unregisterObject(extension, o);
            newExtensions.remove(o);
        }

        // Atomic assignment
        this.extensions = newExtensions.toArray(new StartupExtension[newExtensions.size()]);
    }

    public StartupExtension[] getExtensions() {
        return extensions;
    }

}
