/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.simulator.variable.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.VariantBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.type.Datatype;
import org.simantics.simulator.variable.NodeManager;
import org.simantics.simulator.variable.exceptions.NoSuchNodeException;
import org.simantics.simulator.variable.exceptions.NodeManagerException;

/**
 * AbstractNodeManager gives default implementations to some methods
 * of NodeManager.
 * 
 * @author Hannu Niemist&ouml;
 */
public abstract class AbstractNodeManager<Node> implements NodeManager<Node> {
	

    final static Binding NO_BINDING = new VariantBinding() {

        @Override
        public Object getContent(Object variant, Binding contentBinding) throws BindingException {
            throw new Error();
        }

        @Override
        public Object getContent(Object variant) throws BindingException {
            throw new Error();
        }

        @Override
        public Datatype getContentType(Object variant) throws BindingException {
            throw new Error();
        }

        @Override
        public Binding getContentBinding(Object variant) throws BindingException {
            throw new Error();
        }

        @Override
        public Object create(Binding contentBinding, Object content) throws BindingException {
            throw new Error();
        }

        @Override
        public void setContent(Object variant, Binding contentBinding, Object content) throws BindingException {
            throw new Error();
        }

        @Override
        public boolean isInstance(Object obj) {
            return true;
        }

        @Override
        public void assertInstaceIsValid(Object obj, Set<Object> validInstances) throws BindingException {
            throw new Error();
        }
        
        @Override
        public int compare(Object o1, Object o2) throws org.simantics.databoard.binding.error.RuntimeBindingException {
        	if(o1 == null) {
        		if(o2 == null) {
        			return 0;
        		} else {
        			return  - System.identityHashCode(o2);
        		}
        	} else {
        		if(o2 == null) {
        			return  System.identityHashCode(o1);
        		} else {
        			if(o1.equals(o2)) return 0;
        			return System.identityHashCode(o1) - System.identityHashCode(o2);	
        		}
        	}
        }

    };
    
	@Override
	public List<String> getChildNames(Node node) throws NodeManagerException {
		List<Node> children = getChildren(node);
		ArrayList<String> names = new ArrayList<String>(children.size());
		for(Node child : children)
			names.add(getName(child));
		return names;
	}
	
	@Override
	public List<String> getPropertyNames(Node node) throws NodeManagerException {
		List<Node> properties = getProperties(node);
		ArrayList<String> names = new ArrayList<String>(properties.size());
		for(Node property : properties)
			names.add(getName(property));
		return names;
	}
	
	@Override
	public Object getValue(Node node, String propertyName, Binding binding)
			throws NodeManagerException, BindingException {
		Node property = getProperty(node, propertyName);
		if(property == null)
			throw new NoSuchNodeException("Didn't find a property " + propertyName);
		return getValue(property, binding);
	}
	
	@Override
	public void setValue(Node node, String propertyName, Object value,
			Binding binding) throws NodeManagerException, BindingException {
		Node property = getProperty(node, propertyName);
		if(property == null)
			throw new NoSuchNodeException("Didn't find a property " + propertyName);
		setValue(property, value, binding);
	}
	
	@Override
	public Variant getValue(Node node) throws NodeManagerException {
		Datatype datatype = getDatatype(node);
		Binding binding = datatype != null ? Bindings.getBinding(datatype) : NO_BINDING;
		try {
			Object value = getValue(node, binding);
			if (value instanceof Variant)
				return (Variant)value;
			else
				return new Variant(binding, value);
		} catch (BindingException e) {
			// Shouldn't really happen
			assert(false);
			return new Variant();
		}
	}
	
	@Override
	public Variant getValue(Node node, String propertyName)
			throws NodeManagerException {
		Node property = getProperty(node, propertyName);
		if(property == null)
			throw new NoSuchNodeException("Didn't find a property " + propertyName);
		return getValue(property);
	}
	
    @Override
    public String getPropertyURI(Node parent, Node property) {
        return null;
    }	

}
