package org.simantics.scl.compiler.errors;

import java.util.ArrayList;
import java.util.Collections;

import org.simantics.scl.compiler.common.exceptions.InternalCompilerError;

public class ErrorLog {
    ArrayList<CompilationError> errors = new ArrayList<CompilationError>();
    int errorCount;
    long exceptionPosition = Locations.NO_LOCATION;
    
    public void log(String message) {
        log(new CompilationError(message));
    }
    
    public void log(CompilationError error) {
        errors.add(error);
        if(error.severity != ErrorSeverity.WARNING)
        	++errorCount;
    }
    
    public void log(long locatable, String description) {
        log(new CompilationError(locatable, description));
    }
    
    public void logWarning(long locatable, String description) {
        log(new CompilationError(locatable, description, ErrorSeverity.WARNING));
    }

    public void log(Exception e) {
        long location = Locations.NO_LOCATION;
        if(e instanceof InternalCompilerError)
            location = ((InternalCompilerError)e).location;
        if(location == Locations.NO_LOCATION)
            location = exceptionPosition;
        log(new CompilationError(location, e));
    }
    
    public void log(long location, Exception e) {
        log(new CompilationError(location, e));
    }
    
    public boolean hasNoErrors() {
        return errorCount == 0;
    }
    
    public boolean hasErrors() {
        return errorCount > 0;
    }
    
    public boolean hasErrorsOrWarnings() {
        return !errors.isEmpty();
    }

    public CompilationError[] getErrors() {
        Collections.sort(errors);
        return errors.toArray(new CompilationError[errors.size()]);
    }

    public void setExceptionPosition(long exceptionPosition) {
        if(this.exceptionPosition == Locations.NO_LOCATION)
            this.exceptionPosition = exceptionPosition;
    }

    public String getErrorsAsString() {
        Collections.sort(errors);
        StringBuilder b = new StringBuilder();
        for(CompilationError error : errors)
            b.append(error.description).append('\n');
        return b.toString();
    }
    
    @Override
    public String toString() {
        return getErrorsAsString();
    }

    public int getErrorCount() {
        return errorCount;
    }
}
