/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 *
 * @author Toni Kalajainen
 */
package org.simantics.scenegraph.g2d.events;

import java.util.concurrent.atomic.AtomicBoolean;

import org.simantics.utils.datastructures.prioritystack.PriorityStack;
import org.simantics.utils.threads.CurrentThread;
import org.simantics.utils.threads.IThreadWorkQueue;
import org.simantics.utils.threads.ThreadUtils;

/**
 * @see IEventHandlerStack
 * 
 * @author Toni Kalajainen
 */
public class EventHandlerStack extends PriorityStack<IEventHandler> implements IEventHandlerStack {

    IThreadWorkQueue thread;

    /**
     * Creates a new event handler stack.
     * 
     * Any event handled with this stack is handled in current thread.
     */
    public EventHandlerStack() {
        super(IEventHandler.class);
        this.thread = CurrentThread.getThreadAccess();
    }

    /**
     * Creates a new event handler stack. Any event handled with this stack is
     * handled in the given thread.
     * 
     * @param eventHandlingThread
     */
    public EventHandlerStack(IThreadWorkQueue eventHandlingThread) {
        super(IEventHandler.class);
        this.thread = eventHandlingThread;
    }

    @Override
    public int getEventMask() {
        return EventTypes.AnyMask;
    }

    @Override
    public boolean handleEvent(final Event e) {
        final int eventType = EventTypes.toTypeMask(e);
        if (thread.currentThreadAccess()) {
            // Send event through interactor stack
            IEventHandler list[] = toArray();
            for (int i = list.length - 1; i >= 0; i--)
                if (EventTypes.passes(list[i], eventType) && list[i].handleEvent(e))
                    return true;
            return false;
        } else {
            final AtomicBoolean result = new AtomicBoolean(false);
            Runnable run = new Runnable() {
                @Override
                public void run() {
                    // Send event through interactor stack
                    IEventHandler list[] = toArray();
                    for (int i = list.length - 1; i >= 0; i--)
                        if (EventTypes.passes(list[i], eventType) && list[i].handleEvent(e)) {
                            result.set(true);
                            return;
                        }
                }
            };
            ThreadUtils.syncExec(thread, run);
            return result.get();
        }
    }

}
