package org.simantics.scenegraph.profile.impl;

import java.util.HashSet;
import java.util.Set;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.single.SingleSetSyncListener;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.scenegraph.profile.ProfileEntry;
import org.simantics.scenegraph.profile.common.ProfileObserver;
import org.simantics.utils.datastructures.disposable.IDisposable;

public class ProfileActivationListener extends SingleSetSyncListener<ProfileEntry> {

    final Resource runtime;
    final ProfileObserver observer;
    final IDisposable disposable;
    final Set<ProfileEntry> active = new HashSet<ProfileEntry>();
    
    public ProfileActivationListener(Resource runtime, ProfileObserver observer, IDisposable disposable) {
        this.runtime = runtime;
        this.observer = observer;
        this.disposable = disposable;
    }
    
    @Override
    public boolean start(ReadGraph graph) throws DatabaseException {
        
        if (observer == null || observer.isDisposed())
            return false;
        if (runtime == null)
            return false;

        return true;
        
    }

    @Override
    public void add(ReadGraph graph, ProfileEntry item) throws DatabaseException {
        observer.update();
        item.activate(graph, runtime, observer);
        active.add(item);
        if (DebugPolicy.DEBUG_PROFILE_STYLE_ACTIVATION)
            System.out.println("ACTIVATED PROFILE ENTRY: " + item);
    }

    @Override
    public void remove(ReadGraph graph, ProfileEntry item) throws DatabaseException {
        observer.update();
        active.remove(item);
        if (DebugPolicy.DEBUG_PROFILE_STYLE_ACTIVATION)
            System.out.println("DE-ACTIVATING PROFILE ENTRY: " + item);
        item.deactivate(runtime, observer);
    }

    @Override
    public void finished(ReadGraph graph) throws DatabaseException {
//        observer.update();
    }

    @Override
    public void exception(ReadGraph graph, Throwable t) {
        Logger.defaultLogError(t);
    }

    @Override
    public boolean isDisposed() {
        return disposable.isDisposed();
    }
    
    public void cleanup() {
    	if(!active.isEmpty()) {
    		for(ProfileEntry entry : active) entry.deactivate(runtime, observer);
    		active.clear();
    	}
    }

}