/*******************************************************************************
 * Copyright (c) 2012, 2013 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.internal;

import gnu.trove.map.hash.THashMap;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;

import org.simantics.db.ReadGraph;
import org.simantics.db.WriteGraph;
import org.simantics.objmap.bidirectional.IBidirectionalLinkType;
import org.simantics.objmap.bidirectional.IBidirectionalMapping;
import org.simantics.objmap.bidirectional.IBidirectionalMappingSchema;
import org.simantics.objmap.exceptions.MappingException;

public class BidirectionalMapping<Domain, Range> implements IBidirectionalMapping<Domain, Range> {

    IBidirectionalMappingSchema<Domain, Range> schema;
    THashMap<Domain, BidirectionalLink<Domain, Range>> forwardMap =
            new THashMap<Domain, BidirectionalLink<Domain, Range>>();
    THashMap<Range, BidirectionalLink<Domain, Range>> backwardMap =
            new THashMap<Range, BidirectionalLink<Domain, Range>>();
    
    ArrayList<BidirectionalLink<Domain, Range>> modifiedDomain = new ArrayList<BidirectionalLink<Domain, Range>>();
    ArrayList<BidirectionalLink<Domain, Range>> modifiedRange = new ArrayList<BidirectionalLink<Domain, Range>>();
    
    private void markDomainModified(BidirectionalLink<Domain, Range> link) {
        if(!link.domainModified) {
            link.domainModified = true;
            modifiedDomain.add(link);
        }
    }
    
    private void markRangeModified(BidirectionalLink<Domain, Range> link) {
        if(!link.rangeModified) {
            link.rangeModified = true;
            modifiedRange.add(link);
        }
    }
    
    public BidirectionalMapping(IBidirectionalMappingSchema<Domain, Range> schema) {
        this.schema = schema;
    }

    private BidirectionalLink<Domain, Range> addLink(IBidirectionalLinkType<Domain, Range> linkType, Domain domainElement, Range rangeElement) {
        BidirectionalLink<Domain, Range> link = 
                new BidirectionalLink<Domain, Range>(linkType, domainElement, rangeElement);
        forwardMap.put(domainElement, link);
        backwardMap.put(rangeElement, link);
        return link;
    }
    
    @Override
    public Set<Domain> getDomain() {
        return Collections.unmodifiableSet(forwardMap.keySet());
    }

    @Override
    public Range get(Domain domainElement) {
        BidirectionalLink<Domain, Range> link = forwardMap.get(domainElement);
        if(link == null)
            return null;
        return link.rangeElement;
    }

    @Override
    public Range map(ReadGraph graph, Domain domainElement) throws MappingException {
        Range result = get(domainElement);
        if(result == null) {
            IBidirectionalLinkType<Domain, Range> linkType = 
                    schema.linkTypeOfDomainElement(graph, domainElement);
            Range rangeElement = linkType.createRangeElement(graph, domainElement);
            addLink(linkType, domainElement, rangeElement);
            linkType.createRange(graph, this, domainElement, rangeElement);
        }
        return result;
    }
    
    public Collection<Range> updateRange(ReadGraph graph) throws MappingException {
        ArrayList<Range> updated = new ArrayList<Range>(Math.max(10, modifiedDomain.size())); 
        for(BidirectionalLink<Domain, Range> link : modifiedDomain) {
            link.domainModified = false;
            if(link.linkType.updateRange(graph, this, link.domainElement, link.rangeElement))
                updated.add(link.rangeElement);
        }
        modifiedDomain.clear();
        return updated;        
    }

    @Override
    public Set<Range> getRange() {
        return Collections.unmodifiableSet(backwardMap.keySet());
    }

    @Override
    public Domain inverseGet(Range rangeElement) {
        BidirectionalLink<Domain, Range> link = backwardMap.get(rangeElement);
        if(link == null)
            return null;
        return link.domainElement;
    }

    @Override
    public Domain inverseMap(WriteGraph graph, Range rangeElement)
            throws MappingException {
        Domain result = inverseGet(rangeElement);
        if(result == null) {
            IBidirectionalLinkType<Domain, Range> linkType = 
                    schema.linkTypeOfRangeElement(graph, rangeElement);
            Domain domainElement = linkType.createDomainElement(graph, rangeElement);
            addLink(linkType, domainElement, rangeElement);
            linkType.createDomain(graph, this, domainElement, rangeElement);
        }
        return result;
    }
    
    public Collection<Domain> updateDomain(WriteGraph graph) throws MappingException {
        ArrayList<Domain> updated = new ArrayList<Domain>(Math.max(10, modifiedRange.size())); 
        for(BidirectionalLink<Domain, Range> link : modifiedRange) {
            link.rangeModified = false;
            if(link.linkType.updateDomain(graph, this, link.domainElement, link.rangeElement))
                updated.add(link.domainElement);
        }
        modifiedDomain.clear();
        return updated;        
    }

}
