/*******************************************************************************
 * Copyright (c) 2012, 2013, 2019 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum oy - linked list mapping API and implementation
 *******************************************************************************/
package org.simantics.objmap.graph.schema;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.objmap.bidirectional.IBidirectionalMappingRule;
import org.simantics.objmap.graph.annotations.CompoundRelatedGetValue;
import org.simantics.objmap.graph.annotations.DynamicGraphType;
import org.simantics.objmap.graph.annotations.GraphType;
import org.simantics.objmap.graph.annotations.HasCollectionAdder;
import org.simantics.objmap.graph.annotations.HasCollectionRemover;
import org.simantics.objmap.graph.annotations.HasSetter;
import org.simantics.objmap.graph.annotations.LinkedListGet;
import org.simantics.objmap.graph.annotations.LinkedList;
import org.simantics.objmap.graph.annotations.OptionalRelatedElements;
import org.simantics.objmap.graph.annotations.OrderedElementsGet;
import org.simantics.objmap.graph.annotations.OrderedSetType;
import org.simantics.objmap.graph.annotations.RelatedElement;
import org.simantics.objmap.graph.annotations.RelatedElements;
import org.simantics.objmap.graph.annotations.RelatedElementsGet;
import org.simantics.objmap.graph.annotations.RelatedGetObj;
import org.simantics.objmap.graph.annotations.RelatedGetValue;
import org.simantics.objmap.graph.annotations.RelatedOrderedSetElements;
import org.simantics.objmap.graph.annotations.RelatedValue;
import org.simantics.objmap.graph.annotations.UpdateMethod;
import org.simantics.objmap.graph.annotations.factories.CompoundRelatedGetSetValueRuleFactory;
import org.simantics.objmap.graph.annotations.factories.LinkedListRuleFactory;
import org.simantics.objmap.graph.annotations.factories.LinkedListRuleFactory2;
import org.simantics.objmap.graph.annotations.factories.OptionalRelatedElementsRuleFactory;
import org.simantics.objmap.graph.annotations.factories.OrderedElementsRuleFactory;
import org.simantics.objmap.graph.annotations.factories.RelatedElementRuleFactory;
import org.simantics.objmap.graph.annotations.factories.RelatedElementsRuleFactory;
import org.simantics.objmap.graph.annotations.factories.RelatedElementsRuleFactory2;
import org.simantics.objmap.graph.annotations.factories.RelatedGetSetObjRuleFactory;
import org.simantics.objmap.graph.annotations.factories.RelatedGetSetValueRuleFactory;
import org.simantics.objmap.graph.annotations.factories.RelatedOrderedSetElementsRuleFactory;
import org.simantics.objmap.graph.annotations.factories.RelatedValueRuleFactory;
import org.simantics.objmap.graph.annotations.factories.UpdateMethodFactory;
import org.simantics.objmap.graph.annotations.meta.IsClassRule;
import org.simantics.objmap.graph.annotations.meta.IsCollectionRule;
import org.simantics.objmap.graph.annotations.meta.IsFieldRule;
import org.simantics.objmap.graph.annotations.meta.IsGetSetRule;
import org.simantics.objmap.graph.annotations.meta.IsMethodRule;
import org.simantics.objmap.graph.rules.factory.IClassRuleFactory;
import org.simantics.objmap.graph.rules.factory.ICollectionRuleFactory;
import org.simantics.objmap.graph.rules.factory.IFieldRuleFactory;
import org.simantics.objmap.graph.rules.factory.IGetSetRuleFactory;
import org.simantics.objmap.graph.rules.factory.IMethodRuleFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class MappingSchemas {
    private static final Logger LOGGER = LoggerFactory.getLogger(MappingSchemas.class);

    /**
     * Creates a new SimpleLinkType based on the annotations in the given class.
     * @throws IllegalAccessException 
     * @throws InstantiationException 
     * @see GraphType
     * @see RelatedValue
     */
    public static SimpleLinkType<Object> fromAnnotations(ReadGraph g, Class<?> clazz) throws DatabaseException, InstantiationException, IllegalAccessException {
        GraphType graphType = clazz.getAnnotation(GraphType.class);

        if (graphType != null) {
            ArrayList<IBidirectionalMappingRule<Resource, Object>> rules = new ArrayList<IBidirectionalMappingRule<Resource, Object>>();
            collectRulesFromAnnotations(g, clazz, rules);

            return new SimpleLinkType<Object>(g.getResource(graphType.value()), clazz, rules);  
        }
        DynamicGraphType dynamicType = clazz.getAnnotation(DynamicGraphType.class);
        if (dynamicType != null) {
            ArrayList<IBidirectionalMappingRule<Resource, Object>> rules = new ArrayList<IBidirectionalMappingRule<Resource, Object>>();
            collectRulesFromAnnotations(g, clazz, rules);

            return new DynamicSimpleLinkType<Object>(g.getResource(dynamicType.value()), clazz, rules);
        }
        OrderedSetType orderedSetType = clazz.getAnnotation(OrderedSetType.class);
        if (orderedSetType != null) {
            ArrayList<IBidirectionalMappingRule<Resource, Object>> rules = new ArrayList<IBidirectionalMappingRule<Resource, Object>>();
            collectRulesFromAnnotations(g, clazz, rules);

            return new OrderedSetSimpleLinkType<Object>(g.getResource(orderedSetType.value()), clazz, rules);
        }
        throw new IllegalArgumentException("Class " + clazz.toString() + " does not contain annotations.");
    }

    public static void collectRulesFromAnnotations(ReadGraph g, Class<?> clazz, Collection<IBidirectionalMappingRule<Resource, Object>> rules) throws DatabaseException, InstantiationException, IllegalAccessException {
        Class<?> superclass = clazz.getSuperclass();
        if(superclass != null)
            collectRulesFromAnnotations(g, superclass, rules);

        if (clazz == Object.class)
            return;

        for(Annotation annotation : clazz.getAnnotations()) {

            IsClassRule tag = annotation.annotationType().getAnnotation(IsClassRule.class);
            if(tag!= null) {
                final IClassRuleFactory<Resource, Object> ruleFactory = createClassRule(g, annotation, clazz);
                if (ruleFactory != null)
                    rules.add(ruleFactory.create(g, annotation, clazz));
                else
                    LOGGER.warn("No rule factory found for {}", annotation);
            }
        }

        for(Field f : clazz.getDeclaredFields()) {
            f.setAccessible(true);

            for(Annotation annotation : f.getAnnotations()) {

                IsFieldRule tag = annotation.annotationType().getAnnotation(IsFieldRule.class);
                if(tag != null) {
                    final IFieldRuleFactory<Resource, Object> ruleFactory = createFieldRule(g, annotation, f);
                    if (ruleFactory != null)
                        rules.add(ruleFactory.create(g, annotation, f));
                    else
                        LOGGER.warn("No rule factory found for {}", annotation);
                }
            }
        }

        for(Method m : clazz.getDeclaredMethods()) {
            m.setAccessible(true);

            for(Annotation annotation : m.getAnnotations()) {
                IsMethodRule tag = 
                        annotation.annotationType().getAnnotation(IsMethodRule.class);
                if(tag != null) {
                    final IMethodRuleFactory<Resource, Object> ruleFactory = createMethodRule(g, annotation, m);
                    if (ruleFactory != null)
                        rules.add(ruleFactory.create(g, annotation, m));
                    else
                        LOGGER.warn("No rule factory found for {}", annotation);
                }
            }
        }

        for (Method m : clazz.getDeclaredMethods()) {
            m.setAccessible(true);
            for (Annotation annotation : m.getAnnotations()) {
                Class<? extends Annotation> annotationType = annotation.annotationType();

                IsGetSetRule tag = 
                        annotationType.getAnnotation(IsGetSetRule.class);
                if (tag != null) {

                    HasSetter setterAnnType = annotationType.getAnnotation(HasSetter.class);

                    Class<? extends Annotation> setterAnn = setterAnnType.value();

                    Method getter = m;

                    IGetSetRuleFactory<Resource,Object> ruleFactory = createGetSetRuleFactory(g, annotation, getter);
                    if (ruleFactory != null) {
                        Method setter = null;
    
                        for (Method m2 : clazz.getDeclaredMethods()) {
                            Annotation set = m2.getAnnotation(setterAnn);
                            if (set != null && ruleFactory.isSetter(annotation, set))
                                setter = m2;
                        }
    
                        rules.add(ruleFactory.create(g, annotation, getter, setter));
                    }
                    else {
                        LOGGER.warn("No rule factory found for {}", annotation);
                    }
                }

            }
        }

        for (Method m : clazz.getDeclaredMethods()) {
            m.setAccessible(true);
            for (Annotation annotation : m.getAnnotations()) {
                Class<? extends Annotation> annotationType = annotation.annotationType();

                IsCollectionRule tag = 
                        annotationType.getAnnotation(IsCollectionRule.class);
                if (tag != null) {

                    HasCollectionAdder adderAnnType = annotationType.getAnnotation(HasCollectionAdder.class);
                    HasCollectionRemover removerAnnType = annotationType.getAnnotation(HasCollectionRemover.class);

                    Class<? extends Annotation> adderAnn = adderAnnType.value();
                    Class<? extends Annotation> removerAnn = removerAnnType.value();

                    Method getter = m;

                    ICollectionRuleFactory<Resource,Object> ruleFactory = createCollectionRuleFactory(g, annotation, getter);
                    if (ruleFactory != null) {
                        Method adder = null;
                        Method remover = null;
    
                        for (Method m2 : clazz.getDeclaredMethods()) {
                            Annotation add = m2.getAnnotation(adderAnn);
                            Annotation rem = m2.getAnnotation(removerAnn);
                            if (add != null && ruleFactory.isAdder(annotation, add))
                                adder = m2;
                            if (rem != null && ruleFactory.isRemover(annotation, rem))
                                remover = m2;
                        }
    
                        rules.add(ruleFactory.create(g, annotation, getter,adder,remover));
                    }
                    else {
                        LOGGER.warn("No rule factory found for {}", annotation);
                    }
                }

            }
        }
    }

    public static IClassRuleFactory<Resource, Object> createClassRule(ReadGraph g, Annotation annotation, Class<?> clazz) {
        return null;
    }

    public static IFieldRuleFactory<Resource,Object> createFieldRule(ReadGraph g, Annotation annotation, Field field) {
        if (annotation.annotationType().equals(RelatedElement.class))
            return new RelatedElementRuleFactory<Object>();
        if (annotation.annotationType().equals(RelatedElements.class))
            return new RelatedElementsRuleFactory<Object>();
        if (annotation.annotationType().equals(RelatedValue.class))
            return new RelatedValueRuleFactory<Object>();
        if (annotation.annotationType().equals(OptionalRelatedElements.class))
            return new OptionalRelatedElementsRuleFactory<Object>();
        if (annotation.annotationType().equals(RelatedOrderedSetElements.class))
            return new RelatedOrderedSetElementsRuleFactory<Object>();
        if (annotation.annotationType().equals(LinkedList.class))
            return new LinkedListRuleFactory<Object>();
        return null;
    }

    public static IMethodRuleFactory<Resource, Object> createMethodRule(ReadGraph g, Annotation annotation, Method m) {
        if (annotation.annotationType().equals(UpdateMethod.class))
            return new UpdateMethodFactory<Resource,Object>();
        return null;
    }

    public static IGetSetRuleFactory<Resource,Object> createGetSetRuleFactory(ReadGraph g, Annotation annotation, Method getter) {
        if (annotation.annotationType().equals(RelatedGetValue.class))
            return new RelatedGetSetValueRuleFactory<Object>();
        if (annotation.annotationType().equals(RelatedGetObj.class))
            return new RelatedGetSetObjRuleFactory<Object>();
        if (annotation.annotationType().equals(CompoundRelatedGetValue.class))
            return new CompoundRelatedGetSetValueRuleFactory<Object>();
        return null;
    }

    public static ICollectionRuleFactory<Resource,Object> createCollectionRuleFactory(ReadGraph g, Annotation annotation, Method getter) {
        if (annotation.annotationType().equals(RelatedElementsGet.class))
            return new RelatedElementsRuleFactory2<Object>();
        if (annotation.annotationType().equals(OrderedElementsGet.class))
            return new OrderedElementsRuleFactory<Object>();
        if (annotation.annotationType().equals(LinkedListGet.class))
            return new LinkedListRuleFactory2<>();
        return null;
    }

    /**
     * Creates a new SimpleLinkType based on the annotations in the given class.
     * @throws IllegalAccessException 
     * @throws InstantiationException 
     * @see GraphType
     * @see RelatedValue
     */

    public static AdaptedLinkType<Object> fromAdaptable(ReadGraph g, String type, Class<?> clazz) throws DatabaseException, InstantiationException, IllegalAccessException {


        return new AdaptedLinkType<Object>(g.getResource(type), clazz);    
    }


}
