/*******************************************************************************
 *  Copyright (c) 2023 Association for Decentralized Information Management
 *  in Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.web;

import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Semaphore;

import javax.imageio.ImageIO;

import org.eclipse.core.runtime.IAdaptable;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.DiagramTypeUtils;
import org.simantics.diagram.query.DiagramRequests;
import org.simantics.diagram.symbolcontribution.GroupProxySymbolItem;
import org.simantics.diagram.symbolcontribution.ISymbolProvider;
import org.simantics.diagram.symbolcontribution.IdentifiedObject;
import org.simantics.diagram.symbollibrary.ISymbolGroup;
import org.simantics.diagram.symbollibrary.ISymbolItem;
import org.simantics.g2d.canvas.impl.CanvasContext;
import org.simantics.g2d.diagram.DiagramClass;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.impl.Diagram;
import org.simantics.g2d.diagram.participant.DiagramParticipant;
import org.simantics.g2d.diagram.participant.ElementPainter;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.handler.StaticSymbol;
import org.simantics.g2d.image.DefaultImages;
import org.simantics.g2d.image.Image;
import org.simantics.g2d.participant.TransformUtil;
import org.simantics.scenegraph.g2d.G2DRenderingHints;
import org.simantics.scenegraph.utils.QualityHints;
import org.simantics.scl.runtime.tuple.Tuple2;
import org.simantics.utils.datastructures.hints.IHintContext;
import org.simantics.utils.page.MarginUtils;
import org.simantics.utils.threads.AWTThread;
import org.simantics.utils.threads.IThreadWorkQueue;
import org.simantics.utils.threads.ThreadUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Jussi Koskela
 * @since 1.57.0
 */
public class Symbols {

	private static final Logger LOGGER = LoggerFactory.getLogger(Symbols.class);

	public static List<Tuple2> readSymbols(ReadGraph graph, Resource diagram) throws DatabaseException {
		ISymbolProvider provider = DiagramTypeUtils.readSymbolContributions(graph, diagram);

		List<Tuple2> symbolGroups = new ArrayList<Tuple2>();
		for (ISymbolGroup group : provider.getSymbolGroups()) {
			List<Tuple2> symbols = new ArrayList<Tuple2>();
			symbolGroups.add(new Tuple2(group.getName(), symbols));
			for (ISymbolItem item : group.getItems()) {

				if (item instanceof GroupProxySymbolItem) {
					item = ((GroupProxySymbolItem) item).getSubject();
				}
				if (item instanceof IdentifiedObject) {
					Object id = ((IdentifiedObject) item).getId();
					if (id instanceof IAdaptable) {
						Resource resource = ((IAdaptable) id).getAdapter(Resource.class);
						if (resource != null) {
							symbols.add(new Tuple2(item.getName(), resource));
						}
					}
				}
			}
		}
		return symbolGroups;
	}

	public static byte[] symbolToPNG(Resource symbol) throws DatabaseException, IOException {
		IDiagram diagram = Diagram.spawnNew(DiagramClass.DEFAULT);
		IThreadWorkQueue thread = AWTThread.getThreadAccess();
		CanvasContext ctx = new CanvasContext(thread);

		ctx.add(new TransformUtil());
		ctx.add(new DiagramParticipant());
		ctx.add(new ElementPainter());

		IHintContext hintCtx = ctx.getDefaultHintContext();
		hintCtx.setHint(DiagramHints.KEY_DIAGRAM, diagram);

		ElementClass ec = Simantics.getSession().syncRequest(DiagramRequests.getElementClass(symbol, diagram));
		StaticSymbol ss = ec.getSingleItem(StaticSymbol.class);
		Image source = ss == null ? DefaultImages.UNKNOWN2.get() : ss.getImage();
		source.init(ctx.getCanvasNode());

		final Rectangle2D diagramRect = source.getBounds();
		/*
		 * IElement element = Element.spawnNew(ec); diagram.addElement(element);
		 */

		final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

		final Semaphore done = new Semaphore(0);
		ThreadUtils.asyncExec(AWTThread.getThreadAccess(), new Runnable() {
			@Override
			public void run() {
				try {
					int width = 100;
					int height = 100;

					Rectangle2D controlArea = new Rectangle2D.Double(0, 0, width, height);

					ctx.getSingleItem(TransformUtil.class).fitArea(controlArea, diagramRect, MarginUtils.NO_MARGINS);

					BufferedImage bi = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
					Graphics2D g2 = bi.createGraphics();

					QualityHints.HIGH_QUALITY_HINTS.setQuality(g2);
					g2.setRenderingHint(RenderingHints.KEY_STROKE_CONTROL, RenderingHints.VALUE_STROKE_PURE);

					g2.setTransform(new AffineTransform());
					g2.setClip(new Rectangle2D.Double(0, 0, controlArea.getWidth(), controlArea.getHeight()));

					g2.setRenderingHint(G2DRenderingHints.KEY_CONTROL_BOUNDS,
							new Rectangle2D.Double(0, 0, controlArea.getWidth(), controlArea.getHeight()));

					ctx.getSceneGraph().render(g2);

					try {
						ImageIO.write(bi, "PNG", outputStream);
					} catch (IOException e) {
						LOGGER.error("Failed to write PNG file", e);
					}

				} catch (Exception e) {
					LOGGER.error("Failed to generate symbol image", e);
				} finally {
					done.release();
				}
			}
		});
		try {
			done.acquire();
		} catch (InterruptedException e) {
			LOGGER.error("Symbol rendering was interrupted", e);
		}

		return outputStream.toByteArray();
	}
}
