/*******************************************************************************
 * Copyright (c) 2013, 2022 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - adaption to SCLModuleEditor/TextEditor
 *******************************************************************************/
package org.simantics.modeling.ui.componentTypeEditor;

import java.util.function.Supplier;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ParametrizedRead;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.combinations.Combinators;
import org.simantics.layer0.Layer0;
import org.simantics.scl.ui.editor.SCLSourceViewerConfigurationNew;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.structural2.scl.ComponentTypeScriptPhaseContribution;
import org.simantics.structural2.scl.ComponentTypeScriptPhaseContribution.Phase;
import org.simantics.ui.workbench.IResourceEditorInput;
import org.simantics.ui.workbench.TitleUpdater;
import org.simantics.ui.workbench.ToolTipRequest;
import org.simantics.utils.ui.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Hannu Niemist&ouml;
 * @author Tuukka Lehtonen (extended from SCLModuleEditor)
 */
public class ComponentTypeScriptEditor extends SCLModuleEditor {

    private static final Logger LOGGER = LoggerFactory.getLogger(ComponentTypeScriptEditor.class);

    protected ComponentTypeScriptDocumentProvider docProvider;
    protected Phase[] phases;
    protected String scriptType = "";
    protected int scriptTypeIndex = 0;

    public ComponentTypeScriptEditor() {
        super();
    }

    @Override
    protected void preInitialize() {
        docProvider = new ComponentTypeScriptDocumentProvider(this);
        setDocumentProvider(docProvider);
        SCLSourceViewerConfigurationNew sourceViewerConfiguration = new SCLSourceViewerConfigurationNew(resourceManager);
        setSourceViewerConfiguration(sourceViewerConfiguration);
    }

    protected ParametrizedRead<IResourceEditorInput, Boolean> getInputValidator() {
        // No-op validator that always returns true
        return param -> Combinators.constant(Boolean.TRUE);
    }

    @Override
    protected void updatePartName() {
        setPartName(getEditorInput().getName());

        Session session = Simantics.peekSession();
        if (session != null) {
            Supplier<Boolean> disposedCallback = this::isDisposed;
            session.asyncRequest(
                    new UniqueRead<String>() {
                        @Override
                        public String perform(ReadGraph graph)
                                throws DatabaseException {
                            Layer0 L0 = Layer0.getInstance(graph);
                            StructuralResource2 STR = StructuralResource2.getInstance(graph);
                            Resource script = getResourceInput().getResource();
                            String name = graph.getRelatedValue(script, L0.HasName);
                            Resource componentType = graph.getSingleObject(script, STR.ComponentType_hasScript_Inverse);
                            String ctName = graph.getRelatedValue(componentType, L0.HasName);
                            return ctName + " " + name; //$NON-NLS-1$
                        }
                    },
                    new TitleUpdater(getSite().getShell().getDisplay(), this::setPartName, disposedCallback));
            session.asyncRequest(
                    new ToolTipRequest(getSite().getId(), getResourceInput()),
                    new TitleUpdater(getSite().getShell().getDisplay(), this::setTitleToolTip, disposedCallback));
        }
    }

    private static final Phase[] DEFAULT_EXECUTION_PHASES = {
    };

    @Override
    public void createPartControl(Composite parent) {
        GridLayoutFactory.fillDefaults().applyTo(parent);

        Session session = Simantics.peekSession();
        final Resource script = getResourceInput().getResource();
        this.phases = getPhases(session, script);

        final CCombo combo = new CCombo(parent, SWT.READ_ONLY | SWT.BORDER);
        for(Phase p : phases)
            combo.add(p.label);

        if(session != null) {
            session.asyncRequest(new ReadRequest() {
                @Override
                public void run(ReadGraph graph) throws DatabaseException {
                    StructuralResource2 STR = StructuralResource2.getInstance(graph);
                    String type = graph.getPossibleRelatedValue(script, STR.ComponentTypeScript_type);
                    if(type != null)
                        scriptType = type;
                        combo.getDisplay().asyncExec(() -> {
                            for(int i=0;i<phases.length;++i)
                                if(phases[i].id.equals(type)) {
                                    combo.select(i);
                                    scriptTypeIndex = i;
                                    return;
                                }
                        });
                }
            });
        }
        combo.addSelectionListener(org.eclipse.swt.events.SelectionListener.widgetSelectedAdapter(e -> {
            int id = combo.getSelectionIndex();
            if (id == scriptTypeIndex)
                return;
            if (docProvider.isReadOnly(getEditorInput())) {
                // Return configured selection
                combo.select(scriptTypeIndex);
                return;
            }
            Phase newPhase = phases[id];
            Simantics.getSession().asyncRequest((WriteGraph graph) -> {
                StructuralResource2 STR = StructuralResource2.getInstance(graph);
                String currentType = graph.getPossibleRelatedValue(script, STR.ComponentTypeScript_type);
                if(!newPhase.id.equals(currentType))
                    graph.claimLiteral(script, STR.ComponentTypeScript_type, newPhase.id, Bindings.STRING);
            }, exc -> {
                if (exc == null) {
                    scriptType = newPhase.id;
                    scriptTypeIndex = id;
                } else {
                    ExceptionUtils.logError(exc);
                }
            });
        }));
        GridDataFactory.fillDefaults().grab(true, false).applyTo(combo);

        Composite editorComposite = new Composite(parent, SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, true).applyTo(editorComposite);
        editorComposite.setLayout(new FillLayout());
        super.createPartControl(editorComposite);
    }

    protected Phase[] getPhases(Session session, Resource script) {
        try {
            Phase[] phases = ComponentTypeScriptPhaseContribution.getPhasesForScript(session, script);
            return phases != null ? phases : DEFAULT_EXECUTION_PHASES;
        } catch (DatabaseException ex) {
            LOGGER.error("Failed to get ComponentTypeScriptPhaseContributions for input {}", script, ex);
            return DEFAULT_EXECUTION_PHASES;
        }
    }

}