/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.message;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.Status;

/**
 * A concrete IDetailStatus implementation, suitable either for instantiating or
 * subclassing.
 * <p>
 * This class can be used without OSGi running.
 * </p>
 */
public class DetailStatus extends Status implements IDetailStatus {

    private int severity;

    private String detailedDescription;

    /**
     * Creates a new status object.  The created status has no children.
     *
     * @param severity the severity; one of <code>OK</code>, <code>ERROR</code>,
     *        <code>INFO</code>, <code>WARNING</code>, <code>CANCEL</code>, or
     *        <code>DEBUG</code>
     * @param pluginId the unique identifier of the relevant plug-in
     * @param code the plug-in-specific status code, or <code>OK</code>
     * @param message a human-readable message, localized to the current locale
     * @param exception a low-level exception, or <code>null</code> if not
     *        applicable
     */
    public DetailStatus(int severity, String pluginId, int code, String message, Throwable exception) {
        super(severity, pluginId, code, message, exception);
        setDetailedDescription(null);
    }

    /**
     * Creates a new status object.  The created status has no children.
     *
     * @param severity the severity; one of <code>OK</code>, <code>ERROR</code>,
     *        <code>INFO</code>, <code>WARNING</code>, <code>CANCEL</code>, or
     *        <code>DEBUG</code>
     * @param pluginId the unique identifier of the relevant plug-in
     * @param code the plug-in-specific status code, or <code>OK</code>
     * @param message a human-readable message, localized to the current locale
     * @param detailedDescription a human-readable detailed message, localized
     *        to the current locale
     * @param exception a low-level exception, or <code>null</code> if not
     *        applicable
     */
    public DetailStatus(int severity, String pluginId, int code, String message, String detailedDescription, Throwable exception) {
        super(severity, pluginId, code, message, exception);
        setDetailedDescription(detailedDescription);
    }

    /**
     * Simplified constructor of a new status object; assumes that code is
     * <code>OK</code>. The created status has no children.
     * 
     * @param severity the severity; one of <code>OK</code>, <code>ERROR</code>,
     *        <code>INFO</code>, <code>WARNING</code>, or <code>CANCEL</code>
     * @param pluginId the unique identifier of the relevant plug-in
     * @param message a human-readable message, localized to the current locale
     * @param exception a low-level exception, or <code>null</code> if not
     *        applicable
     */
    public DetailStatus(int severity, String pluginId, String message, Throwable exception) {
        super(severity, pluginId, OK, message, exception);
        setDetailedDescription(null);
    }

    /**
     * Simplified constructor of a new status object; assumes that code is
     * <code>OK</code>. The created status has no children.
     * 
     * @param severity the severity; one of <code>OK</code>, <code>ERROR</code>,
     *        <code>INFO</code>, <code>WARNING</code>, or <code>CANCEL</code>
     * @param pluginId the unique identifier of the relevant plug-in
     * @param message a human-readable message, localized to the current locale
     * @param detailedDescription a human-readable detailed message, localized
     *        to the current locale
     * @param exception a low-level exception, or <code>null</code> if not
     *        applicable
     */
    public DetailStatus(int severity, String pluginId, String message, String detailedDescription, Throwable exception) {
        super(severity, pluginId, OK, message, exception);
        setDetailedDescription(detailedDescription);
    }

    /**
     * Simplified constructor of a new status object; assumes that code is <code>OK</code> and
     * exception is <code>null</code>. The created status has no children.
     *
     * @param severity the severity; one of <code>OK</code>, <code>ERROR</code>,
     * <code>INFO</code>, <code>WARNING</code>,  or <code>CANCEL</code>
     * @param pluginId the unique identifier of the relevant plug-in
     * @param message a human-readable message, localized to the
     *    current locale
     */
    public DetailStatus(int severity, String pluginId, String message) {
        super(severity, pluginId, OK, message, null);
        setDetailedDescription(null);
    }

    /**
     * Simplified constructor of a new status object; assumes that code is
     * <code>OK</code> and exception is <code>null</code>. The created status
     * has no children.
     * 
     * @param severity the severity; one of <code>OK</code>, <code>ERROR</code>,
     *        <code>INFO</code>, <code>WARNING</code>, or <code>CANCEL</code>
     * @param pluginId the unique identifier of the relevant plug-in
     * @param message a human-readable message, localized to the current locale
     * @param detailedDescription a human-readable detailed message, localized
     *        to the current locale
     */
    public DetailStatus(int severity, String pluginId, String message, String detailedDescription) {
        super(severity, pluginId, OK, message, null);
        setDetailedDescription(detailedDescription);
    }


    /**
     * Sets the severity.
     *
     * @param severity the severity; one of <code>OK</code>, <code>ERROR</code>,
     * <code>INFO</code>, <code>WARNING</code>,  or <code>CANCEL</code>
     */
    @Override
    protected void setSeverity(int severity) {
        Assert.isLegal(severity == OK || severity == ERROR || severity == WARNING || severity == INFO || severity == CANCEL || severity == DEBUG);
        this.severity = severity;
    }

    /* (non-Javadoc)
     * @see org.eclipse.core.runtime.Status#getSeverity()
     */
    @Override
    public int getSeverity() {
        return severity;
    }

    /* (non-Javadoc)
     * @see org.simantics.message.IStatus2#getDetailedDescription()
     */
    @Override
    public String getDetailedDescription() {
        return detailedDescription;
    }

    /**
     * @param detailedDescription
     */
    public void setDetailedDescription(String detailedDescription) {
        if (detailedDescription == null)
            this.detailedDescription = ""; //$NON-NLS-1$
        else
            this.detailedDescription = detailedDescription;
    }

    /**
     * Returns a string representation of the status, suitable
     * for debugging purposes only.
     */
    @Override
    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append("Status2 "); //$NON-NLS-1$
        if (severity == OK) {
            buf.append("OK"); //$NON-NLS-1$
        } else if (severity == ERROR) {
            buf.append("ERROR"); //$NON-NLS-1$
        } else if (severity == WARNING) {
            buf.append("WARNING"); //$NON-NLS-1$
        } else if (severity == INFO) {
            buf.append("INFO"); //$NON-NLS-1$
        } else if (severity == CANCEL) {
            buf.append("CANCEL"); //$NON-NLS-1$
        } else if (severity == DEBUG) {
            buf.append("DEBUG"); //$NON-NLS-1$
        } else {
            buf.append("severity="); //$NON-NLS-1$
            buf.append(severity);
        }
        buf.append(": "); //$NON-NLS-1$
        buf.append(getPlugin());
        buf.append(" code="); //$NON-NLS-1$
        buf.append(getCode());
        buf.append(' ');
        buf.append(getMessage());
        buf.append(' ');
        buf.append(getDetailedDescription());
        buf.append(' ');
        buf.append(getException());
        return buf.toString();
    }

}
