package org.simantics.logging;

import java.util.Objects;

import org.slf4j.Logger;

/**
 * @author Tuukka Lehtonen
 * @since 1.47.0
 */
public class LoggerLevel implements Comparable<LoggerLevel> {

	private final Logger logger;
	private final String originalLevel;
	private String loggerName;
	private String level;

	public LoggerLevel(Logger logger, String level) {
		Objects.requireNonNull(logger, "logger");
		Objects.requireNonNull(level, "level");
		this.logger = logger;
		this.originalLevel = level;
		this.loggerName = logger.getName();
		this.level = level;
	}

	public LoggerLevel(String loggerName, String level) {
		Objects.requireNonNull(loggerName, "loggerName");
		Objects.requireNonNull(level, "level");
		this.logger = null;
		this.originalLevel = null;
		this.loggerName = loggerName;
		this.level = level;
	}

	Logger getLogger() {
		return logger;
	}

	public String getName() {
		return loggerName;
	}

	public LoggerLevel setName(String name) {
		if (logger != null)
			throw new UnsupportedOperationException("Cannot rename already created logger " + loggerName + " to " + name);
		Objects.requireNonNull(name, "name");
		this.loggerName = name;
		return this;
	}

	public String getLevel() {
		return level;
	}

	public LoggerLevel setLevel(String level) {
		Objects.requireNonNull(level, "level");
		this.level = level;
		return this;
	}

	public boolean levelChanged() {
		return !Objects.equals(originalLevel, level);
	}

	public boolean isLoggerDefined() {
		return logger != null;
	}

	@Override
	public int hashCode() {
		return Objects.hash(level, logger, loggerName);
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		LoggerLevel other = (LoggerLevel) obj;
		return Objects.equals(level, other.level)
				&& Objects.equals(logger, other.logger)
				&& Objects.equals(loggerName, other.loggerName);
	}

	@Override
	public int compareTo(LoggerLevel o) {
		return loggerName.compareToIgnoreCase(o.loggerName);
	}

}
