/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.layer0.utils.binaryPredicates;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.simantics.db.Resource;
import org.simantics.db.ReadGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.utils.datastructures.Pair;

public class UnionPredicate extends BinaryPredicate {

	Collection<IBinaryPredicate> predicates;

	public UnionPredicate(Collection<IBinaryPredicate> predicates) {
		this.predicates = predicates;
	}
	
	public UnionPredicate(IBinaryPredicate ... predicates) {
		this(Arrays.asList(predicates));
	}

	@Override
	public Collection<Resource> getObjects(ReadGraph g, Resource subject) throws DatabaseException {
		Iterator<IBinaryPredicate> it = predicates.iterator();
		
		while(it.hasNext()) {
			Collection<Resource> result = it.next().getObjects(g, subject);
			if(!result.isEmpty()) {
				while(it.hasNext()) {
					Collection<Resource> temp = it.next().getObjects(g, subject);
					if(!temp.isEmpty()) {
						Set<Resource> merged = new HashSet<Resource>(result.size() + temp.size());
						merged.addAll(result);
						merged.addAll(temp);
						while(it.hasNext())
							merged.addAll(it.next().getObjects(g, subject));
						return merged;
					}
				}
				return result;
			}
		}
		return Collections.emptyList();
	}

	@Override
	public Collection<Pair<Resource, Resource>> getStatements(ReadGraph g) throws DatabaseException {
		Iterator<IBinaryPredicate> it = predicates.iterator();
		
		while(it.hasNext()) {
			Collection<Pair<Resource, Resource>> result = it.next().getStatements(g);
			if(!result.isEmpty()) {
				while(it.hasNext()) {
					Collection<Pair<Resource, Resource>> temp = it.next().getStatements(g);
					if(!temp.isEmpty()) {
						Set<Pair<Resource, Resource>> merged = new HashSet<Pair<Resource, Resource>>(result.size() + temp.size());
						merged.addAll(result);
						merged.addAll(temp);
						while(it.hasNext())
							merged.addAll(it.next().getStatements(g));
						return merged;
					}
				}
				return result;
			}
		}
		return Collections.emptyList();
	}

	@Override
	public Collection<Resource> getSubjects(ReadGraph g, Resource object) throws DatabaseException {
		Iterator<IBinaryPredicate> it = predicates.iterator();
		
		while(it.hasNext()) {
			Collection<Resource> result = it.next().getSubjects(g, object);
			if(!result.isEmpty()) {
				while(it.hasNext()) {
					Collection<Resource> temp = it.next().getSubjects(g, object);
					if(!temp.isEmpty()) {
						Set<Resource> merged = new HashSet<Resource>(result.size() + temp.size());
						merged.addAll(result);
						merged.addAll(temp);
						while(it.hasNext())
							merged.addAll(it.next().getSubjects(g, object));
						return merged;
					}
				}
				return result;
			}
		}
		return Collections.emptyList();
	}

	@Override
	public boolean has(ReadGraph g, Resource subject, Resource object) throws DatabaseException {
		for(IBinaryPredicate pred : predicates)
			if(pred.has(g, subject, object))
				return true;
		return false;
	}

	@Override
	public boolean supportsGetObjects() {
		for(IBinaryPredicate pred : predicates)
			if(!pred.supportsGetObjects())
				return false;
		return true;
	}

	@Override
	public boolean supportsGetStatements() {
		for(IBinaryPredicate pred : predicates)
			if(!pred.supportsGetStatements())
				return false;
		return true;
	}

	@Override
	public boolean supportsGetSubjects() {
		for(IBinaryPredicate pred : predicates)
			if(!pred.supportsGetSubjects())
				return false;
		return true;
	}

	@Override
	public void add(WriteGraph g, Resource subject, Resource object) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void remove(WriteGraph g, Resource subject, Resource object) {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean supportsAdditions() {
		return false;
	}

	@Override
	public boolean supportsRemovals() {
		return false;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((predicates == null) ? 0 : predicates.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		UnionPredicate other = (UnionPredicate) obj;
		if (predicates == null) {
			if (other.predicates != null)
				return false;
		} else if (!predicates.equals(other.predicates))
			return false;
		return true;
	}

}
