/*******************************************************************************
 * Copyright (c) 2007, 2011, 2014 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties.bar;

import java.util.LinkedHashMap;
import java.util.Map;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.layout.LayoutConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.browsing.ui.swt.widgets.Button;
import org.simantics.browsing.ui.swt.widgets.StringPropertyFactory;
import org.simantics.browsing.ui.swt.widgets.StringPropertyModifier;
import org.simantics.browsing.ui.swt.widgets.TrackedCombo;
import org.simantics.browsing.ui.swt.widgets.TrackedText;
import org.simantics.browsing.ui.swt.widgets.impl.ComboModifyListenerImpl;
import org.simantics.browsing.ui.swt.widgets.impl.ReadFactoryImpl;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.jfreechart.ChartPropertyOptions;
import org.simantics.jfreechart.chart.properties.AdjustableTab;
import org.simantics.jfreechart.chart.properties.BooleanPropertyFactory;
import org.simantics.jfreechart.chart.properties.BooleanSelectionListener;
import org.simantics.jfreechart.chart.properties.DoublePropertyFactory2;
import org.simantics.jfreechart.chart.properties.DoublePropertyModifier2;
import org.simantics.jfreechart.chart.properties.DoubleValidator;
import org.simantics.jfreechart.chart.properties.JFreeChartPropertyColorProvider;
import org.simantics.jfreechart.chart.properties.TitleFactory;
import org.simantics.jfreechart.chart.properties.TitleModifier;
import org.simantics.layer0.Layer0;
import org.simantics.layer0.utils.direct.GraphUtils;
import org.simantics.modeling.ui.chart.property.DoublePropertyFactory;
import org.simantics.modeling.ui.chart.property.DoublePropertyModifier;
import org.simantics.sysdyn.JFreeChartResource;


/**
 * General properties of a bar chart
 * @author Teemu Lempinen
 * @author Tuomas Miettinen
 *
 */
public class BarGeneralPropertiesTab extends AdjustableTab {

    private ScrolledComposite sc;
    private Button hgrid, htitle, hlegend;
    private TrackedText name, title, time;
    private TrackedCombo type;
    private TrackedCombo orientation;

    private boolean showTime = true;
    private boolean showFilter = false;
	private Group general;
	private Composite labelColumn1;
	private Composite propertyColumn1;
	private Composite labelColumn2;
	private Composite propertyColumn2;
	private Label labelName;
	private Label labelTitle;
	private Label labelTime;
	private Label labelType;
	private Group typeGroup;
	private Label labelOrientation;
	private Group hideGroup;
	private Label labelUse;
	private Label labelPercent;
	private Group filteringGroup;
	private Button useFilter;
	private TrackedText fraction;
	private Point size;
    
    public BarGeneralPropertiesTab(Object id, int options) {
        super(id);
    	showTime = ((options & ChartPropertyOptions.SHOW_TIME) > 0); 
    	showFilter = ((options & ChartPropertyOptions.SHOW_FILTER) > 0);
    }

    /**
     * 
     * @author Teemu Lempinen
     *
     */
    private class TypeSelectionFactory extends ReadFactoryImpl<Resource, String> {
        @Override
        public String perform(ReadGraph graph, Resource chart) throws DatabaseException {
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);

            Resource plot = graph.syncRequest(new PossibleObjectWithType(chart, l0.ConsistsOf, jfree.CategoryPlot));
            if(plot != null) {
                Resource dataset = graph.syncRequest(new PossibleObjectWithType(plot, l0.ConsistsOf, jfree.CategoryDataset));

                if(dataset != null) {
                    Resource renderer = graph.syncRequest(new PossibleObjectWithType(dataset, jfree.Dataset_renderer, jfree.Renderer));

                    if(renderer != null && graph.isInstanceOf(renderer, jfree.StackedBarRenderer))
                        return "Stacked";
                }
            }
            return "Normal";
        }
    }

    /**
     * RangeItemFactory finds all inexes of a given enumeration 
     * and adds "Sum" and "All" to the returned indexes
     * @author Teemu Lempinen
     *
     */
    private class TypeItemFactory extends ReadFactoryImpl<Resource, Map<String, Object>> {
        @Override
        public Map<String, Object> perform(ReadGraph graph, Resource series) throws DatabaseException {
            LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
            result.put("Normal", "Normal");
            result.put("Stacked", "Stacked");
            return result;
        }
    }

    /**
     * TypeModifyListener for modifying the type of a bar chart 
     * @author Teemu Lempinen
     *
     */
    private class TypeModifyListener  extends ComboModifyListenerImpl<Resource> {
        @Override
        public void applyText(WriteGraph graph, Resource chart, String text) throws DatabaseException {
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);

            Resource plot = graph.syncRequest(new PossibleObjectWithType(chart, l0.ConsistsOf, jfree.CategoryPlot));
            if(plot == null)
                return;

            Resource dataset = graph.syncRequest(new PossibleObjectWithType(plot, l0.ConsistsOf, jfree.CategoryDataset));
            if(dataset == null)
                return;

            graph.deny(dataset, jfree.Dataset_renderer);

            Resource renderer;
            if(text.equals("Stacked"))
                renderer = GraphUtils.create2(graph, jfree.StackedBarRenderer);
            else
                renderer = GraphUtils.create2(graph, jfree.BarRenderer);

            graph.claim(dataset, jfree.Dataset_renderer, renderer);
        }
    }
    
    /**
     * 
     * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
     *
     */
    private class OrientationSelectionFactory extends ReadFactoryImpl<Resource, String> {
        @Override
        public String perform(ReadGraph graph, Resource chart) throws DatabaseException {
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);

            Resource plot = graph.syncRequest(new PossibleObjectWithType(chart, l0.ConsistsOf, jfree.CategoryPlot));
            if(plot != null) {
            	Boolean orientation = graph.getPossibleRelatedValue(plot, jfree.Plot_orientation);
            	if (orientation != null) {
            		if (orientation)
            			return "Horizontal";
            		return "Vertical";			
            	}
                            }
            return "Vertical";
        }
    }

   /**
    * 
    * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
    *
    */
    private class OrientationItemFactory extends ReadFactoryImpl<Resource, Map<String, Object>> {
        @Override
        public Map<String, Object> perform(ReadGraph graph, Resource series) throws DatabaseException {
            LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
            result.put("Vertical", "Vertical");
            result.put("Horizontal", "Horizontal");
            return result;
        }
    }

    /**
     * 
     * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
     *
     */
    private class OrientationModifyListener  extends ComboModifyListenerImpl<Resource> {
        @Override
        public void applyText(WriteGraph graph, Resource chart, String text) throws DatabaseException {
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);

            Resource plot = graph.syncRequest(new PossibleObjectWithType(chart, l0.ConsistsOf, jfree.CategoryPlot));
            if(plot == null)
                return;

           
            if(text.equals("Horizontal"))
                graph.claimLiteral(plot, jfree.Plot_orientation,true);
            else
            	graph.claimLiteral(plot, jfree.Plot_orientation,false);

            
        }
    }

	@Override
	protected void createAndAddControls(Composite body, IWorkbenchSite site,
			ISessionContext context, WidgetSupport support) {
		// Scrolled composite containing all of the properties in this tab
        sc = new ScrolledComposite(body, SWT.NONE | SWT.H_SCROLL | SWT.V_SCROLL);
        sc.setExpandHorizontal(true);
        sc.setExpandVertical(true);

        composite = new Composite(sc, SWT.NONE);

        // General properties
        general = new Group(composite, SWT.NONE);
        general.setText("General");

        // first column: labels
        labelColumn1 = new Composite(general, SWT.NONE);
        
        // second column: name and title
        propertyColumn1 = new Composite(general, SWT.NONE);
        
        // third column: labels
        labelColumn2 = new Composite(general, SWT.NONE);
        
        // fourth column: type and time
        propertyColumn2 = new Composite(general, SWT.NONE);
        
        // Name
        labelName = new Label(labelColumn1, SWT.NONE);
        labelName.setText("Name:");

        name = new org.simantics.browsing.ui.swt.widgets.TrackedText(propertyColumn1, support, SWT.BORDER);
        name.setTextFactory(new StringPropertyFactory(Layer0.URIs.HasLabel));
        name.addModifyListener(new StringPropertyModifier(context, Layer0.URIs.HasLabel));
        name.setColorProvider(new JFreeChartPropertyColorProvider(name.getResourceManager()));

        labelTitle = new Label(labelColumn2, SWT.NONE);
        labelTitle = new Label(propertyColumn2, SWT.NONE);
        
        // Title (Which is different than name)
        labelTitle = new Label(labelColumn1, SWT.NONE);
        labelTitle.setText("Title:");

        title = new org.simantics.browsing.ui.swt.widgets.TrackedText(propertyColumn1, support, SWT.BORDER);
        title.setTextFactory(new TitleFactory());
        title.addModifyListener(new TitleModifier());
        title.setColorProvider(new JFreeChartPropertyColorProvider(name.getResourceManager()));

        if (showTime) {
	        // Time
	        labelTime = new Label(labelColumn2, SWT.NONE);
	        labelTime.setText("Time:");
	        
	        time = new org.simantics.browsing.ui.swt.widgets.TrackedText(propertyColumn2, support, SWT.BORDER);
	        time.setTextFactory(new DoublePropertyFactory(JFreeChartResource.URIs.Chart_time));
	        time.addModifyListener(new DoublePropertyModifier(context, JFreeChartResource.URIs.Chart_time));
	        time.setInputValidator(new DoubleValidator(true));
        time.setColorProvider(new JFreeChartPropertyColorProvider(time.getResourceManager()));
        }
        
        typeGroup = new Group(composite,SWT.NONE);
        typeGroup.setText("Visuals");
        
        labelType = new Label(typeGroup, SWT.NONE);
        labelType.setText("Type:");

        type = new TrackedCombo(typeGroup, support, SWT.BORDER | SWT.READ_ONLY);
        type.addModifyListener(new TypeModifyListener());
        type.setItemFactory(new TypeItemFactory());
        type.setSelectionFactory(new TypeSelectionFactory());
        
        labelOrientation = new Label(typeGroup, SWT.NONE);
        labelOrientation.setText("Orientation:");
        
        orientation = new TrackedCombo(typeGroup, support, SWT.BORDER | SWT.READ_ONLY);
        orientation.addModifyListener(new OrientationModifyListener());
        orientation.setItemFactory(new OrientationItemFactory());
        orientation.setSelectionFactory(new OrientationSelectionFactory());

        
        // Group for hide options
        hideGroup = new Group(composite, SWT.NONE);
        hideGroup.setText("Hide");

        hgrid = new Button(hideGroup, support, SWT.CHECK);
        hgrid.setText("Grid");
        hgrid.setSelectionFactory(new BooleanPropertyFactory(JFreeChartResource.URIs.Plot, JFreeChartResource.URIs.Plot_visibleGrid, true));
        hgrid.addSelectionListener(new BooleanSelectionListener(context, JFreeChartResource.URIs.Plot, JFreeChartResource.URIs.Plot_visibleGrid,true));
        htitle = new Button(hideGroup, support, SWT.CHECK);
        htitle.setText("Title");
        htitle.setSelectionFactory(new BooleanPropertyFactory(JFreeChartResource.URIs.TextTitle, JFreeChartResource.URIs.visible, true));
        htitle.addSelectionListener(new BooleanSelectionListener(context, JFreeChartResource.URIs.TextTitle, JFreeChartResource.URIs.visible,true));
        hlegend = new Button(hideGroup, support, SWT.CHECK);
        hlegend.setText("Legend");
        hlegend.setSelectionFactory(new BooleanPropertyFactory(null, JFreeChartResource.URIs.Chart_visibleLegend, true));
        hlegend.addSelectionListener(new BooleanSelectionListener(context, null, JFreeChartResource.URIs.Chart_visibleLegend,true));

        if (showFilter) {
	        filteringGroup = new Group(composite, SWT.NONE);
	        filteringGroup.setText("Filter");
	        labelUse = new Label(filteringGroup, SWT.NONE);
	        labelUse.setText("Use:");
	        useFilter = new Button(filteringGroup, support, SWT.CHECK);
	        useFilter.setSelectionFactory(new BooleanPropertyFactory(JFreeChartResource.URIs.Plot, JFreeChartResource.URIs.Filter_used, false));
	        useFilter.addSelectionListener(new BooleanSelectionListener(context, JFreeChartResource.URIs.Plot, JFreeChartResource.URIs.Filter_used));
	        labelPercent = new Label(filteringGroup, SWT.NONE);
	        labelPercent.setText("Percent:");
	        fraction = new TrackedText(filteringGroup, support, SWT.BORDER);
	        fraction.setTextFactory(new DoublePropertyFactory2(JFreeChartResource.URIs.Plot,JFreeChartResource.URIs.Filter_fraction));
	        fraction.addModifyListener(new DoublePropertyModifier2(context, JFreeChartResource.URIs.Plot, JFreeChartResource.URIs.Filter_fraction));
	        fraction.setInputValidator(new DoubleValidator(true));
	        fraction.setColorProvider(new JFreeChartPropertyColorProvider(fraction.getResourceManager()));
        }
        
        // Resize scrolled composite
        sc.setContent(composite);
        
	}

	@Override
	protected void createControlLayoutVertical() {
		// Scrolled composite containing all of the properties in this tab
        GridDataFactory.fillDefaults().grab(true, true).applyTo(sc);
        GridLayoutFactory.fillDefaults().applyTo(sc);

        if (showFilter)
        	GridLayoutFactory.fillDefaults().numColumns(1).margins(3, 3).applyTo(composite);
        else
        	GridLayoutFactory.fillDefaults().numColumns(1).margins(3, 3).applyTo(composite);

        // General properties
        GridDataFactory.fillDefaults().grab(true, false).applyTo(general);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(general);

        // first column: labels
        GridDataFactory.fillDefaults().grab(false, true).applyTo(labelColumn1);
        GridLayoutFactory.fillDefaults().applyTo(labelColumn1);
        
        // second column: name and title
        GridDataFactory.fillDefaults().grab(true, true).applyTo(propertyColumn1);
        GridLayoutFactory.fillDefaults().spacing(0, LayoutConstants.getSpacing().y).applyTo(propertyColumn1);
        
        // third column: labels
        GridDataFactory.fillDefaults().grab(false, true).applyTo(labelColumn2);
        GridLayoutFactory.fillDefaults().spacing(0, LayoutConstants.getSpacing().y).applyTo(labelColumn2);
        
        // fourth column: type and time
        GridDataFactory.fillDefaults().grab(false, true).applyTo(propertyColumn2);
        GridLayoutFactory.fillDefaults().applyTo(propertyColumn2);
        
        // Name
        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelName);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(name.getWidget());

        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelTitle);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(title.getWidget());

        if (showTime) {
	        // Time
	        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelTime);
	        
	        GridDataFactory.fillDefaults().align(SWT.FILL, SWT.CENTER).grab(true, false).applyTo(time.getWidget());
        }
        
        GridDataFactory.fillDefaults().applyTo(typeGroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(typeGroup);
        
        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelType);

        GridDataFactory.fillDefaults().applyTo(type.getWidget());
        
        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelOrientation);
        
        GridDataFactory.fillDefaults().applyTo(type.getWidget());

        // Group for hide options
        GridDataFactory.fillDefaults().applyTo(hideGroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(hideGroup);

        if (showFilter) {
	        GridDataFactory.fillDefaults().applyTo(filteringGroup);
	        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(filteringGroup);
	        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelUse);
	        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelPercent);
	        GridDataFactory.fillDefaults().align(SWT.FILL, SWT.CENTER).grab(true, false).applyTo(fraction.getWidget());
        }
        
        size = composite.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        sc.setMinSize(size);
	}

	@Override
	protected void createControlLayoutHorizontal(boolean wideScreen) {
		// Scrolled composite containing all of the properties in this tab
        GridDataFactory.fillDefaults().grab(true, true).applyTo(sc);
        GridLayoutFactory.fillDefaults().applyTo(sc);

        if (showFilter)
        	GridLayoutFactory.fillDefaults().numColumns(4).margins(3, 3).applyTo(composite);
        else
        	GridLayoutFactory.fillDefaults().numColumns(3).margins(3, 3).applyTo(composite);

        // General properties
        GridDataFactory.fillDefaults().grab(true, false).applyTo(general);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(4).applyTo(general);

        // first column: labels
        GridDataFactory.fillDefaults().grab(false, true).applyTo(labelColumn1);
        GridLayoutFactory.fillDefaults().applyTo(labelColumn1);
        
        // second column: name and title
        GridDataFactory.fillDefaults().grab(true, true).applyTo(propertyColumn1);
        GridLayoutFactory.fillDefaults().spacing(0, LayoutConstants.getSpacing().y).applyTo(propertyColumn1);
        
        // third column: labels
        GridDataFactory.fillDefaults().grab(false, true).applyTo(labelColumn2);
        GridLayoutFactory.fillDefaults().spacing(0, LayoutConstants.getSpacing().y).applyTo(labelColumn2);
        
        // fourth column: type and time
        GridDataFactory.fillDefaults().grab(false, true).applyTo(propertyColumn2);
        GridLayoutFactory.fillDefaults().applyTo(propertyColumn2);
        
        // Name
        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelName);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(name.getWidget());

        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelTitle);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(title.getWidget());

        if (showTime) {
	        // Time
	        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelTime);
	        
	        GridDataFactory.fillDefaults().align(SWT.FILL, SWT.CENTER).grab(true, false).applyTo(time.getWidget());
        }
        
        GridDataFactory.fillDefaults().applyTo(typeGroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(typeGroup);
        
        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelType);

        GridDataFactory.fillDefaults().applyTo(type.getWidget());
        
        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelOrientation);
        
        GridDataFactory.fillDefaults().applyTo(type.getWidget());

        // Group for hide options
        GridDataFactory.fillDefaults().applyTo(hideGroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(hideGroup);

        if (showFilter) {
	        GridDataFactory.fillDefaults().applyTo(filteringGroup);
	        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(filteringGroup);
	        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelUse);
	        GridDataFactory.fillDefaults().grab(false, true).align(SWT.END, SWT.CENTER).applyTo(labelPercent);
	        GridDataFactory.fillDefaults().align(SWT.FILL, SWT.CENTER).grab(true, false).applyTo(fraction.getWidget());
        }
        
        size = composite.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        sc.setMinSize(size);
	}

}
