/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.element;

import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;

import org.jfree.chart.JFreeChart;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.DiagramUtils;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.SceneGraphNodeKey;
import org.simantics.g2d.element.handler.HandleMouseEvent;
import org.simantics.g2d.element.handler.InternalSize;
import org.simantics.g2d.element.handler.PropertySetter;
import org.simantics.g2d.element.handler.SceneGraph;
import org.simantics.scenegraph.Node;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.g2d.events.MouseEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseEnterEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseExitEvent;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintListener;
import org.simantics.utils.datastructures.hints.IHintObservable;

/**
 * Chart scenegraph for chart elements in diagrams
 * @author Teemu Lempinen
 *
 */
public class ChartSceneGraph  implements SceneGraph, HandleMouseEvent, InternalSize {
    private static final long serialVersionUID = 1875762898776996989L;

    public static final Key KEY_SG_NODE = new SceneGraphNodeKey(Node.class, "CHART_SG_NODE");
    public static final Key KEY_SG_SELECTION_NODE = new SceneGraphNodeKey(Node.class, "CHART_SG_SELECTION_NODE");

    protected IHintListener hoverHintListener;

    public ChartSceneGraph() {
    }

    /**
     * Expands bounds a little to allow selecting and grabbing a chart element
     * from outside the chart graphics
     */
    @Override
    public Rectangle2D getBounds(IElement e, Rectangle2D s) {
        if (s==null) s = new Rectangle2D.Double();

        IG2DNode node = (IG2DNode)e.getHint(KEY_SG_NODE);
        AffineTransform at = (AffineTransform)e.getHint(ElementHints.KEY_TRANSFORM);
        if(at != null && node != null && node.getBoundsInLocal() != null) {
            Shape shape = node.getBoundsInLocal();
            Rectangle2D r = shape.getBounds2D();
            double scaleX = at.getScaleX();
            double scaleY = at.getScaleY();
            ChartNode.expand(r, 1 * scaleX, 1 * scaleY);
            shape = r;
            s.setFrame(shape.getBounds2D());
        } else {
            s.setFrame((Rectangle2D)e.getHint(ElementHints.KEY_BOUNDS));
        }
        return s;
    }

    @Override
    public void cleanup(IElement e) {
        if(hoverHintListener != null)
            e.removeHintListener(hoverHintListener);

        Node node = e.removeHint(KEY_SG_NODE);
        if (node != null)
            node.remove();
    }

    @Override
    public void init(final IElement e, G2DParentNode parent) {
        ChartNode node = e.getHint(KEY_SG_NODE);
        if(node == null) {
            // Create a new chart node
            node = parent.getOrCreateNode("chart_"+e.hashCode(), ChartNode.class);

            Rectangle2D bounds = (Rectangle2D)e.getHint(ElementHints.KEY_BOUNDS);
            if(bounds == null) {
                bounds = new Rectangle2D.Double(-40, -40, 80, 80);
                e.setHint(ElementHints.KEY_BOUNDS, bounds);
            }
            node.setBounds(bounds);

            JFreeChart chart = e.getHint(ChartElementFactory.KEY_CHART);
            if(chart != null)
                node.setChart(chart);

            // Add a resize listener for updating bounds information to graph after resizing
            node.setResizeListener(new ResizeListener() {

                @Override
                public void elementResized(Rectangle2D newBounds) {
                    e.setHint(ElementHints.KEY_BOUNDS, newBounds);
                    IDiagram diagram = ElementUtils.getDiagram(e);
                    DiagramUtils.synchronizeHintsToBackend(diagram, e);
                }
            });

            e.setHint(KEY_SG_NODE, node);
        }

        // Hover listening
        hoverHintListener = new IHintListener() {
            @Override
            public void hintRemoved(IHintObservable sender, Key key, Object oldValue) {

            }

            @Override
            public void hintChanged(IHintObservable sender, Key key, Object oldValue, Object newValue) {
                if(key == ElementHints.KEY_HOVER) {
                    IElement e = (IElement)sender;
                    ChartNode name = (ChartNode) e.getHint(KEY_SG_NODE);
                    if (name != null)
                        name.setHover(Boolean.TRUE.equals(e.getHint(ElementHints.KEY_HOVER)));
                }
            }
        };
        e.addHintListener(hoverHintListener);

        update(e);
    }

    public void update(IElement e) {
        PropertySetter setter = e.getElementClass().getSingleItem(PropertySetter.class);
        setter.syncPropertiesToNode(e);
    }

    
    @Override
    public boolean handleMouseEvent(IElement e, ICanvasContext ctx, MouseEvent me) {
        if (me instanceof MouseEnterEvent) {
            e.setHint(ElementHints.KEY_HOVER, true);
        } else if (me instanceof MouseExitEvent) {
            e.setHint(ElementHints.KEY_HOVER, false);
        }
        return false;
    }
}
