package org.simantics.jfreechart.chart;

/**
 * Filters CategoryDataset by creating "Other" item for filtered data.
 * 
 * Filtering uses sum of absolute values as a comparison point and compares absolute values.
 * If negative values are filtered, their absolute value is added to the others category.  
 * 
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
import java.util.List;

import org.jfree.data.category.DefaultCategoryDataset;
import org.jfree.data.general.AbstractDataset;
import org.jfree.data.general.DatasetChangeEvent;
import org.jfree.data.general.DatasetChangeListener;

@SuppressWarnings("rawtypes")
public class FilteringCategoryDataset extends AbstractDataset implements org.jfree.data.category.CategoryDataset, FilteredDataset, DatasetChangeListener{
	
	private static final long serialVersionUID = -4955124650051030544L;
	
	org.jfree.data.category.CategoryDataset original;
	DefaultCategoryDataset filtered;
	org.jfree.data.category.CategoryDataset used;
	
	boolean filterRows = true;
	boolean filtering = true;
	double filterFraction = 0.05;
	private Comparable other;
	
	public FilteringCategoryDataset(org.jfree.data.category.CategoryDataset dataset, Comparable other) {
		this.original = dataset;
		this.filtered = new DefaultCategoryDataset();
		this.other = other;
		this.used = filtered;
		this.original.addChangeListener(this);
		updateFiltered();
	}
	
	@Override
	public boolean isFiltering() {
		return filtering;
	}
	
	@Override
	public void setFiltering(boolean filtering) {
		this.filtering = filtering;
		if (filtering)
			used = filtered;
		else
			used = original;
		fireDatasetChanged();
	}
	
	public void setFilterFraction(double filterFraction) {
		this.filterFraction = filterFraction;
	}
	
	public double getFilterFraction() {
		return filterFraction;
	}
	
	/**
	 * Filter rows or columns.
	 * @param filterRows
	 */
	public void setFilterRows(boolean filterRows) {
		this.filterRows = filterRows;
	}
	
	public boolean isFilterRows() {
		return filterRows;
	}
	
	public void updateFiltered() {
		filtered.clear();
		if (filterRows) {
			for (Object column : original.getColumnKeys()) {
				Double total = 0.0;
				Double other = 0.0;
				for (Object row : original.getRowKeys()) {
					Number value =  original.getValue((Comparable) row, (Comparable)column);
					if (value != null)
						total+=Math.abs(value.doubleValue());
				}
				total *= filterFraction;
				for (Object row : original.getRowKeys()) {
					Number value = original.getValue((Comparable) row, (Comparable)column);
					if (value == null)
						continue;
					if (Math.abs(value.doubleValue()) > total) {
						filtered.addValue(value, (Comparable) row, (Comparable)column);
					} else {
						other += value.doubleValue(); 
					}
				}
				if (Math.abs(other) > 0.0) {
					filtered.addValue(other, this.other, (Comparable)column);
				}
			}
		} else {
			for (Object row : original.getRowKeys()) {
				Double total = 0.0;
				Double other = 0.0;
				for (Object column : original.getColumnKeys()) {
					Number value = original.getValue((Comparable) row, (Comparable)column);
					if (value != null)
						total += Math.abs(value.doubleValue());
				}
				total *= filterFraction;
				for (Object column : original.getColumnKeys()) {
					Number value = original.getValue((Comparable) row, (Comparable)column);
					if (value == null)
						continue;
					if (Math.abs(value.doubleValue()) > total) {
						filtered.addValue(value, (Comparable) row, (Comparable)column);
					} else {
						other += value.doubleValue(); 
					}
				}
				if (Math.abs(other) > 0.0) {
					filtered.addValue(other, (Comparable)row, this.other);
				}
			}
		}
		fireDatasetChanged();
	}
	
	@Override
	public int getColumnCount() {
		return used.getColumnCount();
	}
	
	@Override
	public int getRowCount() {
		return used.getRowCount();
	}
	
	@Override
	public Number getValue(Comparable rowKey, Comparable columnKey) {
		return used.getValue(rowKey, columnKey);
	}
	
	@Override
	public Number getValue(int row, int column) {
		return used.getValue(row, column);
	}
	
	@Override
	public List getColumnKeys() {
		return used.getColumnKeys();
	}	
	
	@Override
	public Comparable getColumnKey(int column) {
		return used.getColumnKey(column);
	}
	
	@Override
	public List getRowKeys() {
		return used.getRowKeys();
	}
	
	@Override
	public Comparable getRowKey(int row) {
		return used.getRowKey(row);
	}
	
	@Override
	public int getRowIndex(Comparable key) {
		return used.getRowIndex(key);
	}
	
	@Override
	public int getColumnIndex(Comparable key) {
		return used.getColumnIndex(key);
	}
	
	public Comparable getOther() {
		return other;
	}
	
	@Override
	public void datasetChanged(DatasetChangeEvent event) {
		// bypass change events from the original dataset.
		fireDatasetChanged();
	}
	

}
