/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridLayout;

import javax.swing.JPanel;

import org.eclipse.swt.widgets.Composite;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.simantics.Simantics;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncListener;
import org.simantics.db.request.Read;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.threads.AWTThread;
import org.simantics.utils.threads.ThreadUtils;
import org.simantics.utils.ui.SWTAWTComponent;

/**
 * Composite containing a single chart defined by a JFreeChart.Chart
 * 
 * Similar to ChartComposite, but uses SWTAWTComponent as a base implementation. 
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 * 
 */
public class ChartComposite2 extends SWTAWTComponent {

    private JPanel jPanel;
    private ChartPanel chartPanel;
    private IJFreeChart chart;

    public ChartComposite2(Composite parent, final String chartResourceURI, int style) {
        super(parent, style);
        try {
            Resource chartResource = Simantics.getSession().syncRequest(
                    (Read<Resource>) graph -> graph.getPossibleResource(chartResourceURI));
            if(chartResource != null)
                CreateContent(chartResource);
        } catch (DatabaseException e) {
            e.printStackTrace();
        }
        syncPopulate();
    }

    public ChartComposite2(Composite parent, final Resource chartResource, int style) {
        super(parent, style);
        CreateContent(chartResource);
        syncPopulate();
    }

    protected ChartComposite2(Composite parent, int style) {
        super(parent, style);
    }

    @Override
    protected Component createSwingComponent() {
        jPanel = new JPanel();
        jPanel.setLayout(new GridLayout(1, 1));
        if (chartPanel != null)
            jPanel.add(chartPanel);
        jPanel.doLayout();
        return jPanel;
    }

    protected void setPanel(final ChartPanel panel) {
        ThreadUtils.asyncExec(AWTThread.getThreadAccess(), () -> {
            if (jPanel == null) {
                chartPanel = panel;
            } else {
                jPanel.removeAll();
                chartPanel = panel;
                jPanel.add(chartPanel, BorderLayout.CENTER);
                jPanel.add(chartPanel);
                jPanel.doLayout();
            }
        });
    }

    public ChartPanel getChartPanel() {
        return chartPanel;
    }

    public IJFreeChart getChart() {
        return chart;
    }

    @Override
    public void doDispose() {
        if (chart != null) {
            chart.dispose();
            chart = null;
        }
        super.doDispose();
    }

    protected Read<IJFreeChart> getChartQuery(final Resource chartResource) {
        return new ChartRead(chartResource);
    }

    /**
     * This query does not implement equals or hashCode by purpose. The result is not reusable.
     * 
     * @author luukkainen
     */
    protected static class ChartRead implements Read<IJFreeChart> {
        private Resource chartResource;

        public ChartRead(Resource chartResource) {
            this.chartResource = chartResource;
        }

        @Override
        public IJFreeChart perform(ReadGraph graph) throws DatabaseException {
            // Adapt chartResource to a chart (XY, pie, bar, ...)
            if(graph.isInstanceOf(chartResource, JFreeChartResource.getInstance(graph).Chart)) {
                return graph.adaptUnique(chartResource, IJFreeChart.class);
            } else {
                return null;
            }
        }
    }

    /**
     * Creates and displays the chart defined in chartResource
     * @param chartResource
     */
    protected void CreateContent(final Resource chartResource) {

        // Add a listener displaying the contents of the chart. Chart is re-drawn if the definition changes
        Simantics.getSession().asyncRequest(getChartQuery(chartResource) , new AsyncListener<IJFreeChart>() {

            @Override
            public boolean isDisposed() {
                return ChartComposite2.this.isDisposed();
            }

            @Override
            public void execute(AsyncReadGraph graph, IJFreeChart chart) {
                if(ChartComposite2.this.chart != null) {
                    ChartComposite2.this.chart.dispose();
                    ChartComposite2.this.chart = null;
                }

                if(chart == null) {
                    return;
                }
                if (ChartComposite2.this.isDisposed()) {
                    chart.dispose();
                    return;
                }
                ChartComposite2.this.chart = chart;
                JFreeChart jfreeChart = chart.getChart();

                ChartPanel panel = new ChartPanel(jfreeChart,
                        ChartPanel.DEFAULT_WIDTH,
                        ChartPanel.DEFAULT_HEIGHT,
                        ChartPanel.DEFAULT_MINIMUM_DRAW_WIDTH,
                        ChartPanel.DEFAULT_MINIMUM_DRAW_HEIGHT,
                        ChartPanel.DEFAULT_MAXIMUM_DRAW_WIDTH, 
                        ChartPanel.DEFAULT_MAXIMUM_DRAW_HEIGHT,
                        false,
                        false, true, true, true, true);
                setPanel(panel);
            }

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                throwable.printStackTrace();
            }
        });
    }

}
