/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSource;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.Read;
import org.simantics.jfreechart.internal.Activator;
import org.simantics.layer0.Layer0;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.ui.dnd.LocalObjectTransfer;
import org.simantics.utils.datastructures.Pair;

/**
 * Header of a chart element in {@link ChartPanel}. Only this header is
 * shown if a chart is minimized. If a chart is expanded, this header is added
 * to the charts {@link ChartPanelElement}. 
 * 
 * @author Teemu Lempinen
 *
 */
public class ChartPanelHeader extends Composite {

    public static int HEADER_MINIMUM_WIDTH = 250;
    private ChartPanelElement element;
    private Resource resource;
    private Label name;
    private Canvas iconCanvas;
    private Image icon;
    private ToolItem minimize, remove;
    private Color defaultColor, darker, evenDarker;
    private Image gradientBackgroundImage, borderImage;

    private static ImageDescriptor closeDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/close.gif"));
    private static Image closeImage = closeDescriptor.createImage();

    private static ImageDescriptor minimizeDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/minimize.gif"));
    private static Image minimizeImage = minimizeDescriptor.createImage();

    private static ImageDescriptor maximizeDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/maximize.gif"));
    private static Image maximizeImage = maximizeDescriptor.createImage();

    private static ImageDescriptor lineChartDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/chart_line_light.png"));
    private static Image lineChartImage = lineChartDescriptor.createImage();

    private static ImageDescriptor barChartDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/chart_bar_light.png"));
    private static Image barChartImage = barChartDescriptor.createImage();

    private static ImageDescriptor pieChartDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/chart_pie_light.png"));
    private static Image pieChartImage = pieChartDescriptor.createImage();


    /**
     * Chart panel header with minimize and close buttons.
     * 
     * @param parent The composite where the header is added
     * @param panel The {@link ChartPanel} containing the header
     * @param name The name of the chart
     * @param style he Style of the created chart element
     */
    public ChartPanelHeader(Composite c, ChartPanelElement element, Resource chartResource, int style) {
        super(c, style);
        this.resource = chartResource;
        this.element = element;

        GridLayoutFactory.fillDefaults().margins(3, 0).numColumns(3).applyTo(this);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(this);

        // Colors

        // Chart icon
        iconCanvas = new Canvas (this, SWT.NONE);
        GridDataFactory.fillDefaults().align(SWT.CENTER, SWT.CENTER).hint(16, 16).applyTo(iconCanvas);
        iconCanvas.addPaintListener (new PaintListener() {

            @Override
            public void paintControl(PaintEvent e) {
                if(icon != null)
                    e.gc.drawImage (icon, 0, 0);                
            }
        });

        // Label for the chart name (also minimize/expand)
        name = new Label(this, SWT.NONE);

        try {
            // name updater
            Pair<String, Image> result = Simantics.getSession().syncRequest(new Read<Pair<String, Image>>() {

                @Override
                public Pair<String, Image> perform(ReadGraph graph) throws DatabaseException {
                    JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
                    Layer0 l0 = Layer0.getInstance(graph);
                    String label = NameUtils.getSafeLabel(graph, resource);
                    Image image = null;
                    Resource plot = graph.syncRequest(new PossibleObjectWithType(resource, l0.ConsistsOf, jfree.Plot));
                    if(plot != null) {
                        if(graph.isInstanceOf(plot, jfree.CategoryPlot))
                            image = barChartImage;
                        else if(graph.isInstanceOf(plot, jfree.PiePlot))
                            image = pieChartImage;
                        else
                            image = lineChartImage;
                    }
                    return new Pair<String, Image>(label, image);
                }

            }, new Listener<Pair<String, Image>>() {

                @Override
                public void execute(final Pair<String, Image> result) {
                    if(result == null)
                        return;

                    name.getDisplay().asyncExec(new Runnable() {

                        @Override
                        public void run() {
                            if(!name.isDisposed() && result.first != null)
                                name.setText(result.first);

                            if(!iconCanvas.isDisposed() && result.second != null) {
                                icon = result.second;
                                iconCanvas.redraw();
                                ChartPanelHeader.this.layout();
                            }
                        }
                    });
                }

                @Override
                public void exception(Throwable t) {
                    t.printStackTrace();
                }

                @Override
                public boolean isDisposed() {
                    return name.isDisposed();
                }

            });
            name.setText(result.first);
        } catch (DatabaseException e) {
            e.printStackTrace();
            name.setText("No label");
        }
        GridDataFactory.fillDefaults().grab(true, false).applyTo(name);

        ToolBar toolbar = new ToolBar(this, SWT.FLAT);
        // item for minimizing/expanding chart
        minimize = new ToolItem(toolbar, SWT.PUSH);
        minimize.addSelectionListener(new MinimizeListener());
        if(isMinimized()) {
            minimize.setToolTipText("Expand");
            minimize.setImage(maximizeImage);
        } else {
            minimize.setToolTipText("Minimize");
            minimize.setImage(minimizeImage);
        }

        // item for closing/removing the chart
        remove = new ToolItem(toolbar, SWT.PUSH);
        remove.setImage(closeImage);
        remove.addSelectionListener(new RemoveChartListener());
        remove.setToolTipText("Remove");


        /* ********************************
         * DnD 
         * ********************************/

        // Allow data to be copied or moved from the drag source
        int operations = DND.DROP_MOVE;
        source = new DragSource(name, operations);

        // Provide data in Text format
        Transfer[] types = new Transfer[] {  LocalObjectTransfer.getTransfer() };
        source.setTransfer(types);
        dragSourceListener = new DragSourceListener() {

            @Override
            public void dragStart(DragSourceEvent event) {
                if(name.isDisposed())
                    event.doit = false;
                event.detail = DND.DROP_LINK;

            }

            @Override
            public void dragSetData(DragSourceEvent event) {
                event.data = new StructuredSelection(resource);
            }

            @Override
            public void dragFinished(DragSourceEvent event) {
            }
        };  
        source.addDragListener(dragSourceListener);

        name.addDisposeListener(new DisposeListener() {

            @Override
            public void widgetDisposed(DisposeEvent e) {
                if(dragSourceListener != null && source != null && !source.isDisposed()) {
                    source.removeDragListener(dragSourceListener);
                }                
            }
        });
        this.setBackgroundImage(getGradientBackgroundImage());
        this.setBackgroundMode(SWT.INHERIT_FORCE);

        this.addListener(SWT.MouseEnter, new EnterListener());
        this.addListener(SWT.MouseExit, new ExitListener());

        for(Control child : this.getChildren()) {
            child.addListener(SWT.MouseEnter, new EnterListener());
            child.addListener(SWT.MouseExit, new ExitListener());

        }
    }

    private class EnterListener implements  org.eclipse.swt.widgets.Listener {
        public void handleEvent(Event event) {
            ChartPanelHeader.this.setBackgroundImage(getHighlightedGradientBackgroundImage());
        }
    }

    private class ExitListener implements  org.eclipse.swt.widgets.Listener {
        public void handleEvent(Event event) {
            ChartPanelHeader.this.setBackgroundImage(getGradientBackgroundImage());
        }
    }

    private void createColors() {
        if(defaultColor == null) {
            defaultColor = getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND);
            try {
                defaultColor = new Color(getDisplay(), defaultColor.getRed() + 500, defaultColor.getGreen() + 10, defaultColor.getBlue() + 10);
            } catch (IllegalArgumentException e) {
                // Do nothing, use the default color
            }
        }

        if(darker == null) {
            try {
                darker = new Color(getDisplay(), defaultColor.getRed() - 30, defaultColor.getGreen() - 30, defaultColor.getBlue() - 30);
            } catch (IllegalArgumentException e) {
                // Do nothing, use the default color
                darker = defaultColor;
            }
        }
        
        if(evenDarker == null) {
            try {
                evenDarker = new Color(getDisplay(), defaultColor.getRed() - 50, defaultColor.getGreen() - 50, defaultColor.getBlue() - 50);
            } catch (IllegalArgumentException e) {
                // Do nothing, use the default color
                evenDarker = defaultColor;
            }
        }

    }

    private Image getHighlightedGradientBackgroundImage() {
        createColors();
        this.layout();
        Point size = this.getSize();

        borderImage = new Image(this.getDisplay(), 1, Math.max(1, size.y));
        GC gc = new GC(borderImage);
        gc.setForeground(defaultColor);
        gc.setBackground(evenDarker);
        gc.fillGradientRectangle(0, 0, 1, size.y, true);
        gc.dispose();

        return borderImage;
    }

    private Image getGradientBackgroundImage() {
        createColors();
        this.layout();
        Point size = this.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        if(gradientBackgroundImage == null) {
            gradientBackgroundImage = new Image(this.getDisplay(), 1, Math.max(1, size.y));
            GC gc = new GC(gradientBackgroundImage);
            gc.setForeground(defaultColor);
            gc.setBackground(darker);
            gc.fillGradientRectangle(0, 0, 1, size.y, true);
            gc.dispose();
        }

        return gradientBackgroundImage;
    }

    private DragSourceListener dragSourceListener;
    private DragSource source;

    /**
     * Return true if this element is minimized, false if expanded
     * @return true if this element is minimized, false if expanded
     */
    private boolean isMinimized() {
        return element.isMinimized();
    }

    /**
     * Listener to minimize chart button. Expands and minimizes 
     * the chart of this header.
     * 
     * @author Teemu Lempinen
     *
     */
    private class MinimizeListener implements SelectionListener {
        @Override
        public void widgetSelected(SelectionEvent e) {
            if(ChartPanelHeader.this.isDisposed())
                return;

            element.toggleMinimize(true);

            if(!name.isDisposed() && !minimize.isDisposed()) {
                if(isMinimized()) {
                    minimize.setToolTipText("Expand");
                } else {
                    minimize.setToolTipText("Minimize");
                }
            }            
        }

        @Override
        public void widgetDefaultSelected(SelectionEvent e) {
            widgetSelected(e);
        }

    }

    /**
     * Listener for removing this chart from the chart panel.
     * 
     * @author Teemu Lempinen
     *
     */
    private class RemoveChartListener implements SelectionListener {
        @Override
        public void widgetSelected(SelectionEvent e) {
            if(!ChartPanelHeader.this.isDisposed()) {
                element.remove();
            }
        }

        @Override
        public void widgetDefaultSelected(SelectionEvent e) {
            widgetSelected(e);
        }

    }
}
