/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart;

import java.util.ArrayList;
import java.util.LinkedHashMap;

import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.DropTargetAdapter;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.ViewPart;
import org.simantics.Simantics;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncListener;
import org.simantics.db.request.Read;
import org.simantics.jfreechart.internal.Activator;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.ui.dnd.LocalObjectTransfer;
import org.simantics.utils.RunnableWithObject;

/**
 * Chart panel displays multiple charts in a single view. The view can be oriented 
 * vertically or horizontally, the default is vertical. Charts can be added, removed 
 * minimized or expanded. The order of the charts can be changed by dragging the charts.
 * 
 * @author Teemu Lempinen
 *
 */
public class ChartPanel extends ViewPart {

    private Composite body;
    private ScrolledComposite sc;

    private IDialogSettings settings;
    private LinkedHashMap<Resource, ChartPanelElement> charts;
    private ArrayList<Resource> minimizedResources;

    private ArrayList<ChartPanelElement> chartElements;

    public static final String CHART_PANEL_SETTINGS = "CHART_PANEL_SETTINGS";
    public static final String CHARTS = "CHART_PANEL_CHARTS";
    public static final String MINIMIZED_CHARTS = "CHART_PANEL_MINIMIZED_CHARTS";
    public static final String CHART_PANEL_ORIENTATION = "CHART_PANEL_ORIENTATION";

    public static final String CHART_PANEL_VERTICAL = "CHART_PANEL_ORIENTATION_VERTICAL";
    public static final String CHART_PANEL_HORIZONTAL = "CHART_PANEL_ORIENTATION_HORIZONTAL";

    private boolean vertical = true;

    /**
     * Initialize the view. Load charts that have previously been open (if there are any).
     */
    @Override
    public void init(IViewSite site, IMemento memento) throws PartInitException {
        super.init(site, memento);

        minimizedResources = new ArrayList<Resource>();

        settings = Activator.getDefault().getDialogSettings().getSection(CHART_PANEL_SETTINGS);

        // Initialize settings if there are no settings
        if (settings == null) {
            settings = Activator.getDefault().getDialogSettings().addNewSection(CHART_PANEL_SETTINGS);
        }

        if(settings.getArray(CHARTS) == null) {
            String[] chartUris = new String[] {};
            settings.put(CHARTS, chartUris);
        }

        if(settings.getArray(MINIMIZED_CHARTS) == null) {
            String[] minimizedChartUris = new String[] {};
            settings.put(MINIMIZED_CHARTS, minimizedChartUris);
        }

        // initialize chart lists
        charts = new LinkedHashMap<Resource, ChartPanelElement>();

        // add chart resources to chart lists from settings
        try {
            Simantics.getSession().syncRequest(new ReadRequest() {

                @Override
                public void run(ReadGraph graph) throws DatabaseException {
                    JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
                    Resource chart = null;
                    String[] chartURIs = settings.getArray(CHARTS);
                    for(String uri : chartURIs) {
                        chart = graph.getPossibleResource(uri);
                        if(chart != null && graph.isInstanceOf(chart, jfree.Chart)) {
                            charts.put(chart, null);
                            setChartExistingListener(chart);
                        }
                    }

                    String[] minimizedUris = settings.getArray(MINIMIZED_CHARTS);
                    for(String uri : minimizedUris) {
                        chart = graph.getPossibleResource(uri);
                        if(chart != null && graph.isInstanceOf(chart, jfree.Chart)) {
                            minimizedResources.add(chart);
                        } 
                    }
                }
            });
        } catch (DatabaseException e1) {
            e1.printStackTrace();
        }

        // set the orientation of the panel
        String orientation = settings.get(CHART_PANEL_ORIENTATION);
        if(CHART_PANEL_VERTICAL.equals(orientation))
            this.vertical = true;
        else if(CHART_PANEL_HORIZONTAL.equals(orientation))
            this.vertical = false;

    }

    /**
     * Create a scrolled composite that will contain all the charts, then call the actual 
     * content creator.
     */
    @Override
    public void createPartControl(Composite parent) {
        sc = new ScrolledComposite(parent, SWT.NONE | SWT.H_SCROLL | SWT.V_SCROLL);
        GridLayoutFactory.fillDefaults().spacing(0, 0).applyTo(sc);
        GridDataFactory.fillDefaults().grab(true, true).applyTo(sc);
        sc.setExpandHorizontal(true);
        sc.setExpandVertical(true);
        sc.getVerticalBar().setIncrement(sc.getVerticalBar().getIncrement()*3);
        sc.getHorizontalBar().setIncrement(sc.getHorizontalBar().getIncrement()*3);

        body = new Composite(sc, SWT.NONE);
        GridLayoutFactory.fillDefaults().margins(3, 0).spacing(0, 0).applyTo(body);
        GridDataFactory.fillDefaults().grab(true, true).applyTo(body);

        sc.setContent(body);
        createContents();
        
        setupDropTarget();

    }
    
    /**
     * Creates the contents of this chart panel.
     * Removes all old contents before creating new content
     */
    private void createContents() {
        chartElements = new ArrayList<ChartPanelElement>();

        for(Control child : body.getChildren()) {
            child.dispose();
        }
        
        // Set the initial layout
        ElementContainer elementHolder;
        for(Resource e : charts.keySet()) {
            elementHolder = new ElementContainer(body, SWT.NONE);
            elementHolder.setBackground(new Color(elementHolder.getDisplay(), 255, 0, 0));
            ChartPanelElement element = new ChartPanelElement(elementHolder, this, e, SWT.NONE);
            elementHolder.setLayout(GridLayoutFactory.copyLayout((GridLayout)element.getLayout()));
            chartElements.add(element);
            charts.put(e, element);
            if(minimizedResources.contains(e)) {
                element.toggleMinimize();
            }
        }

        elementHolder = new ElementContainer(body, SWT.NONE);
        elementHolder.setBackground(new Color(elementHolder.getDisplay(), 0, 255, 0));
        ChartPanelElement element = new ChartPanelElement(elementHolder, this, null, SWT.NONE); // Last element is empty -> only the separator
        elementHolder.setLayout(GridLayoutFactory.copyLayout((GridLayout)element.getLayout()));
        chartElements.add(element);

        layout();
        saveState();

    }

    /**
     * Lays out this panel (the body composite)
     */
    public void layout() {
        if(vertical) {
            GridLayoutFactory.fillDefaults().spacing(0, 0).applyTo(body);
            GridDataFactory.fillDefaults().grab(true, true).applyTo(body);
        } else {
            // Need to calculate horizontal elements for gridLayout
            int chartPanels = chartElements.size();
            GridLayoutFactory.fillDefaults().spacing(0, 0).numColumns(chartPanels).applyTo(body);
            GridDataFactory.fillDefaults().grab(true, true).applyTo(body);
        }
        body.layout();
        sc.setMinSize(body.computeSize(SWT.DEFAULT, SWT.DEFAULT));
    }

    @Override
    public void setFocus() {
        if(!sc.isDisposed())
            sc.setFocus();
    }

    @Override
    public void saveState(IMemento memento) {
        super.saveState(memento);
        saveState();
    }

    /**
     * Save the current state of the view to IDialogSettings 
     */
    public void saveState() {
        try {
            Simantics.getSession().syncRequest(new ReadRequest() {

                @Override
                public void run(ReadGraph graph) throws DatabaseException {
                    if (settings != null) {
                        String[] uris = new String[chartElements.size() - 1];
                        ArrayList<String> minimized = new ArrayList<String>();
                        minimizedResources.clear();
                        for(int i = 0; i < uris.length; i++) {
                            ChartPanelElement e = chartElements.get(i);
                            Resource r = e.getResource();
                            if(r != null) {
                                uris[i] = graph.getURI(r);
                                if(e.isMinimized()) {
                                    minimized.add(uris[i]);
                                    minimizedResources.add(r);
                                }
                            } else {
                                uris[i] = "";
                            }
                        }
                        settings.put(CHARTS, uris);
                        if(!minimized.isEmpty())
                            settings.put(MINIMIZED_CHARTS, minimized.toArray(new String[minimized.size()]));
                        else
                            settings.put(MINIMIZED_CHARTS, new String[0]);

                        if(vertical)
                            settings.put(CHART_PANEL_ORIENTATION, CHART_PANEL_VERTICAL);
                        else
                            settings.put(CHART_PANEL_ORIENTATION, CHART_PANEL_HORIZONTAL);
                    }
                }
            });
        } catch (DatabaseException e) {
            e.printStackTrace();
        }
    }

    /**
     * Set the orientation for this chart panel.
     * 
     * @param orientation Orientation (ChartPanel.CHART_PANEL_VERTICAL or ChartPanel.CHART_PANEL_HORIZONTAL)
     */
    public void setOrientation(String orientation) {
        if(CHART_PANEL_VERTICAL.equals(orientation))
            this.vertical = true;
        else {
            this.vertical = false;
        }
        createContents();
    }

    /**
     * Removes a chart from this panel
     * 
     * @param chart The chart to be removed
     */
    public void removeChart(Resource chart) {
        ChartPanelElement element = charts.get(chart);
        chartElements.remove(element);
        element.getParent().dispose();
        charts.remove(chart);
        minimizedResources.remove(chart);
        saveState();
        layout();
    }

    /**
     * Sets up drag-scrolling for a scrolled composite 
     * 
     * @param control
     */
    protected void setupDropTarget() {
        DropTarget target = new DropTarget(sc, DND.DROP_MOVE);
        target.setTransfer(new Transfer[] { LocalObjectTransfer.getTransfer() });

        target.addDropListener(new DropTargetAdapter() {

            private int activeMargin = 20;
            private int moveAmount = 1;

            @Override
            public void dragOver(DropTargetEvent event) { 
                Point original = sc.getOrigin();
                Point origin = sc.getOrigin();
                Point pointer = sc.toControl(event.x, event.y);
                Rectangle bounds = sc.getBounds();

                if(pointer.y < activeMargin)
                    origin.y = origin.y - moveAmount;
                else if(bounds.height - pointer.y < activeMargin)
                    origin.y = origin.y + moveAmount;
                if(pointer.x < activeMargin)
                    origin.x = origin.x - moveAmount;
                else if(bounds.width - pointer.x < activeMargin)
                    origin.x = origin.x + moveAmount;

                if(origin != original) {
                    sc.setOrigin (origin.x, origin.y);
                    sc.redraw();
                }
            }

        });
        
        DropTarget target2 = new DropTarget(body, DND.DROP_COPY | DND.DROP_MOVE);
        target2.setTransfer(new Transfer[] {  LocalObjectTransfer.getTransfer() });
        target2.addDropListener(new ChartDropTarget(body, null, this));

    }

    /**
     * Is the panel vertically oriented
     * @return Is the panel vertically oriented
     */
    public boolean isVertical() {
        return vertical;
    }
    
    /**
     * Adds chart after given element. If element == null, adds chart to the top.
     * 
     * @param element To which place the chart will be placed. (null allowed)
     */
    public void addChart(Resource chartResource, ChartPanelElement element) {
        addChart(chartResource, element, true);
    }

    /**
     * Adds chart after given element. If element == null, adds chart to the top.
     * 
     * @param element To which place the chart will be placed. (null allowed)
     * @param layout refresh layout. use with vertical layout. 
     */
    public void addChart(Resource chartResource, ChartPanelElement element, boolean layout) {
        if(element == null)
            element = chartElements.get(chartElements.size() - 1);
        int index = chartElements.indexOf(element);
        if(index >= 0) {
            ChartPanelElement e = chartElements.get(index);
            ChartPanelElement newElement;

            if(charts.containsKey(chartResource)) {
                // Old element being moved to a new place
                newElement = charts.get(chartResource);
                int oldIndex = chartElements.indexOf(newElement);
                if(newElement.equals(element) || oldIndex == index - 1)
                    return; // Not moving anywhere, do nothing
                Composite oldParent = newElement.getParent();
                newElement.setParent(e.getParent());
                oldParent.dispose();
                if(oldIndex < index)
                    index--;
                chartElements.remove(newElement);
            } else {
                newElement = new ChartPanelElement(e.getParent(), this, chartResource, SWT.NONE);
            }

            // Add a new chart element to the location of the old element
            chartElements.add(index, newElement);
            charts.put(chartResource, newElement);

            ElementContainer elementHolder;
            // Move elements back after index
            for(int i = index + 1 /*indexes after the new element*/; i < chartElements.size(); i++) {
                e = chartElements.get(i);
                if(i == chartElements.size() - 1) {
                    // last element (the empty element) element to a new container
                    elementHolder = new ElementContainer(body, SWT.NONE);
                    elementHolder.setBackground(new Color(elementHolder.getDisplay(), 0, 0, 255));
                    elementHolder.setLayout(GridLayoutFactory.copyLayout((GridLayout)e.getLayout()));
                    e.setParent(elementHolder);
                } else {
                    // element to the next elements container
                    elementHolder = (ElementContainer)chartElements.get(i + 1).getParent();
                    e.setParent(elementHolder);
                }
            }
            
            layout();
            saveState();
        }
    }

    /**
     * Set a listener to listen if the chart resource has been removed.
     * If the resource has been removed, close also the chart element in this panel.
     * 
     * @param chart Listened chart resource
     */
    private void setChartExistingListener(final Resource chart) {
        Simantics.getSession().asyncRequest(new Read<Boolean>() {

            @Override
            public Boolean perform(ReadGraph graph) throws DatabaseException {
                return graph.hasStatement(chart);
            }
        }, new AsyncListener<Boolean>() {

            boolean disposed = false;

            @Override
            public void execute(AsyncReadGraph graph, Boolean result) {
                if(result != null && result == false && body != null && !body.isDisposed()) {
                    body.getDisplay().asyncExec(new RunnableWithObject(chart){
                        public void run() {
                            removeChart((Resource)getObject());
                        }
                    }) ;
                    disposed = true;
                }
            }

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                throwable.printStackTrace();
            }

            @Override
            public boolean isDisposed() {
                return !disposed && !charts.containsKey(chart);
            }
        });
    }

    public ArrayList<ChartPanelElement> getElements() {
        return chartElements;
    }
}
