/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.graphviz;

import java.io.PrintStream;

/**
 * A node of a graph.
 * 
 * @author Hannu Niemist
 */
public class Node extends AbstractIdentifiableGraphPart {

    public Node(IGraph graph) {
        this(graph, "");
    }

    public Node(IGraph graph, String label) {
        super(graph);
        setLabel(label);
    }

    @Override
    public void write(PrintStream s) {
        s.print(id);
        writeAttributes(s);
    }

    /**
     * Text label attached to objects. If a node's shape is record, then the label can have a special format which describes the record layout.
     */
    public void setLabel(String label) {
        set("label", label);
    }

    /**
     * <p>
     * Basic drawing color for graphics, not text. For the latter, use the
     * fontcolor attribute. For edges, the value can either be a single color or
     * a colorList. In the latter case, the edge is drawn using parallel splines
     * or lines, one for each color in the list, in the order given. The head
     * arrow, if any, is drawn using the first color in the list, and the tail
     * arrow, if any, the second color. This supports the common case of drawing
     * opposing edges, but using parallel splines instead of separately routed
     * multiedges.
     * </p>
     * <p>Supports arbitrary RGB(A) colors in format #xxxxxx, where
     * each x is a hex digit. Supports also color names in x11 color chart.</p>
     */
    public void setColor(String color) {
        set("color", color);
    }
    
    /**
     * Color used to fill the background of a node or cluster assuming style=filled. If fillcolor is not defined, color is used. (For clusters, if color is not defined, bgcolor is used.) If this is not defined, the default is used, except for shape=point or when the output format is MIF, which use black by default. Note that a cluster inherits the root graph's attributes if defined. Thus, if the root graph has defined a fillcolor, this will override a color or bgcolor attribute set for the cluster.
     */
    public void setFillColor(String color) {
        set("fillcolor", color);
    }
    
    
    /**
     * Color used for text.
     */
    public void setFontColor(String color) {
        set("fontcolor", color);
    }
    
    /**
     * Set style for node or edge. For cluster subgraph, if "filled", the cluster box's background is filled.
     */
    public void setStyle(String style) {
        set("style", style);
    }

    /**
     * <p>
     * Set the shape of a node.
     * </p>
     * <p>
     * Alternatives:
     * 
     * 
     * <tt>box polygon ellipse circle point egg triangle plaintext diamond trapezium parallelogram house pentagon hexagon septagon octagon doublecircle doubleoctagon tripleoctagon invtriangle invtrapezium invhouse Mdiamond Msquare Mcircle rect rectangle none note tab box3d component</tt>
     * </p>
     */
    public void setShape(String shape) {
        set("shape", shape);
    }
    
    /**
     * Width of node, in inches. This is taken as the initial, minimum width of the node. If fixedsize is true, this will be the final width of the node. Otherwise, if the node label requires more width to fit, the node's width will be increased to contain the label. Note also that, if the output format is dot, the value given to width will be the final value.
     */
    public void setWidth(double width) {
        set("width", Double.toString(width));
    }
    
    /**
     * Height of node, in inches. This is taken as the initial, minimum height of the node. If fixedsize is true, this will be the final height of the node. Otherwise, if the node label requires more height to fit, the node's height will be increased to contain the label. Note also that, if the output format is dot, the value given to height will be the final value.
     */
    public void setHeight(double width) {
        set("height", Double.toString(width));
    }
    
    /**
     * If true, the node size is specified by the values of the width and height attributes only and is not expanded to contain the text label.
     */
    public void setFixedSize(boolean fixedSize) {
        set("fixedsize", Boolean.toString(fixedSize));
    }
    
    public Identifiable getPort(final String portName) {
        return new Identifiable() {

            @Override
            public String getId() {
                return id + ":" + portName;
            }

            @Override
            public IGraph getParent() {
                return parent;
            }
            
        };
    }

}
