/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.graphfile.adapters;

import java.io.File;
import java.io.IOException;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.ActionFactory;
import org.simantics.db.request.Read;
import org.simantics.graphfile.Activator;
import org.simantics.graphfile.ontology.GraphFileResource;
import org.simantics.graphfile.util.GraphFileUtil;
import org.simantics.layer0.Layer0;
import org.simantics.utils.ui.dialogs.ShowError;

/**
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class ExportSystemResourcesActionFactory implements ActionFactory{
	
	enum ResourceType{FILE,FOLDER,UNKNOW};
	
	@Override
	public Runnable create(Object _target) {
		final Resource target = (Resource)_target;
		return new Runnable() {
			
			@Override
			public void run() {
				try {
					Shell shell= Display.getCurrent().getActiveShell();
					
					ResourceType type = getResourceType(target);
				
					if (type == ResourceType.UNKNOW)
						return;
					
					String name = getName(target);
					
					String exportName = null;
					
					if (type == ResourceType.FILE) {
						FileDialog dialog = new FileDialog(shell, SWT.SAVE);
						dialog.setFileName(name);
						exportName = dialog.open();
					} else {
						DirectoryDialog dialog = new DirectoryDialog(shell, SWT.SAVE);
						exportName = dialog.open();
					}
					if (exportName == null)
						return;
					
					final File targetFile = new File(exportName);
					
					if (type == ResourceType.FILE) {
						exportFile(shell, targetFile, target);
						
					} else {
						exportFolder(shell, targetFile, target);
					}
				
				} catch (Exception e) {
					Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Failed to export file/folder to disk: " + target , e));
					ShowError.showError("Error", e.getMessage(), e);
				} 
				
			}
		};
	}
	
	private String getName(final Resource resource) throws DatabaseException {
		return Simantics.getSession().syncRequest(new Read<String>() {
			
			public String perform(ReadGraph graph) throws DatabaseException {	
				Layer0 l0 = Layer0.getInstance(graph);
				return graph.getPossibleRelatedValue(resource, l0.HasName);
			 }
		});
	}
	
	private ResourceType getResourceType(final Resource resource) throws DatabaseException {
		return Simantics.getSession().syncRequest(new Read<ResourceType>() {
			
			public ResourceType perform(ReadGraph graph) throws DatabaseException {	
				GraphFileResource gf = GraphFileResource.getInstance(graph);
				if (graph.isInstanceOf(resource, gf.File))
					return ResourceType.FILE;
				if (graph.isInstanceOf(resource, gf.Folder))
					return ResourceType.FOLDER;
				return ResourceType.UNKNOW;
			}
		});
	}
	
	private void exportFile(Shell shell, final File targetFile, final Resource target) throws Exception{
		if (targetFile.exists()) {
			if (!targetFile.isFile()) {
        		MessageDialog.openError(shell, "File Problem", "Output target is not a file " + targetFile.getAbsolutePath());
                return;
        	}
			boolean ok = MessageDialog.openConfirm(shell, "Overwrite", "A file by the name " + targetFile.getAbsolutePath() + " contains files.\n\nDo you want to overwrite the files?");
    		if (!ok) {
    			return;
    		}
    		if (!targetFile.delete()) {
    			MessageDialog.openError(shell, "Delete Problem", "Could not overwrite previously existing file " + targetFile.getAbsolutePath());
                return;
    		}
		}
		Simantics.getSession().syncRequest(new ReadRequest() {
			
			@Override
			public void run(ReadGraph graph) throws DatabaseException {
				try {
					GraphFileUtil.writeDataToFile(graph, target , targetFile);
				} catch (IOException e) {
					throw new DatabaseException(e);
				}
				
			}
		});
	}
	
	private void exportFolder(Shell shell, final File targetFile, final Resource target) throws Exception{
		if (targetFile.exists()) {
			if (!targetFile.isDirectory()) {
				MessageDialog.openError(shell, "Folder Problem", "Output target is not a folder " + targetFile.getAbsolutePath());
				return;
			}
			String files[] = targetFile.list();
			if (files.length > 0) {
				boolean ok = MessageDialog.openConfirm(shell, "Overwrite", "A folder by the name " + targetFile.getAbsolutePath() + " contains files.\n\nDo you want to overwrite the files?");
				if (!ok) {
					return;
				}
				GraphFileUtil.clearDirectoryStructure(targetFile);
			}

		} else {
			if (!targetFile.mkdir()) {
				MessageDialog.openError(shell, "Folder Problem", "Could not create new folder " + targetFile);
				return;
			}
		}
		Simantics.getSession().syncRequest(new ReadRequest() {
			
			@Override
			public void run(ReadGraph graph) throws DatabaseException {
				try {
					GraphFileUtil.writeFolderToDisk(graph, target, targetFile);
				} catch (IOException e) {
					throw new DatabaseException(e);
				}
				
			}
		});
	}

}
