/*******************************************************************************
 * Copyright (c) 2007, 2022 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - Gitlab #795
 *******************************************************************************/
/*
 *
 * @author Toni Kalajainen
 */
package org.simantics.g2d.event.adapter;

import java.awt.geom.Point2D;

import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.MouseTrackListener;
import org.eclipse.swt.events.MouseWheelListener;
import org.eclipse.swt.graphics.Point;
import org.simantics.scenegraph.g2d.events.EventDebugPolicy;
import org.simantics.scenegraph.g2d.events.IEventHandler;
import org.simantics.scenegraph.g2d.events.IEventQueue;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseButtonPressedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseButtonReleasedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseDoubleClickedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseEnterEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseExitEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseMovedEvent;
import org.simantics.scenegraph.g2d.events.adapter.AbstractEventAdapter;
import org.simantics.utils.ui.SWTDPIUtil;

public class SWTMouseEventAdapter extends AbstractEventAdapter implements MouseListener, MouseMoveListener, MouseTrackListener, MouseWheelListener {

	/** Mouse id of the default mouse */
	public static final int MOUSE_ID = 0;
	
    private long [] pressTime = new long[5];
    
	int buttonStatus = 0;

	/**
	 * create new adapter
	 * @param sender the sender field in the events
	 * @param delegator the target of the adapted events
	 */
	public SWTMouseEventAdapter(Object sender, IEventHandler delegator) {
		super(sender, delegator);
	}

	/**
	 * create new adapter
	 * @param sender the sender field in the events
	 * @param queue
	 */
	public SWTMouseEventAdapter(Object sender, IEventQueue queue) {
		super(sender, queue);
	}
	
	private Point2D getControlPosition(MouseEvent e)
	{
		return new Point2D.Double(SWTDPIUtil.upscaleSwt(e.x), SWTDPIUtil.upscaleSwt(e.y));
	}
	
	private Point2D getScreenPosition(MouseEvent e)
	{
		Point p = SWTDPIUtil.upscaleSwt(e.display.getCursorLocation());
		return new Point2D.Double(p.x, p.y);
		
	}
	
	private int getMouseButton(MouseEvent e)
	{
		switch (e.button) {
			case 1: return 1; // LEFT
			case 2: return 3; // RIGHT
			case 3: return 2; // MIDDLE
			case 4: return 4; // BUTTON4
			case 5: return 5; // BUTTON5
		}
		return e.button;
	}
	
	static private int getStateMask(MouseEvent e) {
	    return SWTKeyEventAdapter.adaptSWTModifierToAWTModifier(e.stateMask);
	}

	@Override
	public void mouseDoubleClick(MouseEvent e) {
		if (EventDebugPolicy.SWT_MOUSE_EVENT_ADAPTION)
			System.out.println("SWT mouse double clicked: " + e);
		handleEvent(new MouseDoubleClickedEvent(
				sender, e.time & 0xffffffff, MOUSE_ID, buttonStatus, getStateMask(e), getMouseButton(e), getControlPosition(e), getScreenPosition(e)
				));
	}

	@Override
	public void mouseDown(MouseEvent e) {
		if (EventDebugPolicy.SWT_MOUSE_EVENT_ADAPTION)
			System.out.println("SWT mouse down: " + e);
		buttonStatus |= 1 << e.button;		
		if (e.button<=pressTime.length) pressTime[e.button-1] = e.time & 0xffffffff;		
		handleEvent(new MouseButtonPressedEvent(
				sender, e.time & 0xffffffff, MOUSE_ID, buttonStatus, getStateMask(e), getMouseButton(e), getControlPosition(e), getScreenPosition(e)
				));
	}

	@Override
	public void mouseUp(MouseEvent e) {
		if (EventDebugPolicy.SWT_MOUSE_EVENT_ADAPTION)
			System.out.println("SWT mouse up: " + e);
		buttonStatus &=~ (1<<e.button);
		long holdTime = Long.MAX_VALUE;
		if (e.button<=pressTime.length)
			holdTime = (e.time & 0xffffffff) - pressTime[e.button-1];
		
		handleEvent(new MouseButtonReleasedEvent(
				sender, e.time & 0xffffffff, MOUSE_ID, buttonStatus, getStateMask(e), getMouseButton(e), holdTime, getControlPosition(e), getScreenPosition(e)
				));
	}

	@Override
	public void mouseMove(MouseEvent e) {
		if (EventDebugPolicy.SWT_MOUSE_EVENT_ADAPTION)
			System.out.println("SWT mouse move: " + e);
		MouseMovedEvent mme = new MouseMovedEvent(
			sender, e.time & 0xffffffff, MOUSE_ID, buttonStatus, getStateMask(e), getControlPosition(e), getScreenPosition(e));
		handleEvent(mme);
	}

	@Override
	public void mouseEnter(MouseEvent e) {
		if (EventDebugPolicy.SWT_MOUSE_EVENT_ADAPTION)
			System.out.println("SWT mouse enter: " + e);
		handleEvent(new MouseEnterEvent(
				sender, e.time & 0xffffffff, MOUSE_ID, buttonStatus, getStateMask(e), getControlPosition(e), getScreenPosition(e)
				));
	}

	@Override
	public void mouseExit(MouseEvent e) {
		if (EventDebugPolicy.SWT_MOUSE_EVENT_ADAPTION)
			System.out.println("SWT mouse exit: " + e);
		handleEvent(new MouseExitEvent(
				sender, e.time & 0xffffffff, MOUSE_ID, buttonStatus, getStateMask(e), getControlPosition(e), getScreenPosition(e)
				));
	}

	@Override
	public void mouseHover(MouseEvent e) {
		if (EventDebugPolicy.SWT_MOUSE_EVENT_ADAPTION)
			System.out.println("SWT mouse hover: " + e);
	}

	@Override
	public void mouseScrolled(MouseEvent e) {
		if (EventDebugPolicy.SWT_MOUSE_EVENT_ADAPTION)
			System.out.println("SWT mouse scrolled: " + e);
		// #795: AWT already produces mouse wheel events so lets not emit
		// MouseWheelMovedEvents from SWT at all to avoid producing duplicate
		// events.
		/*
		handleEvent(new MouseWheelMovedEvent(
				sender, e.time & 0xffffffff, MOUSE_ID, buttonStatus, 
				getStateMask(e),
				getControlPosition(e), getScreenPosition(e), 
				MouseWheelMovedEvent.WHEEL_UNIT_SCROLL,
				0,
				e.count
				));
		*/
	}

}
