/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.element.impl;

import java.util.Map;

import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.ElementAdapter;
import org.simantics.g2d.element.handler.LifeCycle;
import org.simantics.utils.datastructures.hints.HintContext;

/**
 * @author Toni Kalajainen
 */
public class Element extends HintContext implements IElement {

    /**
     * Spawn brand new element to the world.
     * The element is initialized.
     * 
     * @param clazz
     * @return
     */
    public static IElement clone(IElement cloneFrom)
    {
        IElement e = new Element(cloneFrom.getElementClass());
        e.setHints(cloneFrom.getHints());
        return e;
    }

    /**
     * Spawn brand new element to the world.
     * The element is initialized.
     * 
     * @param clazz
     * @return
     */
    public static IElement spawnNew(ElementClass clazz)
    {
        assert(clazz!=null);
        IElement e = new Element(clazz);
        Element.fireCreated(e);
        return e;
    }

    /**
     * Instantiate old element back into the world. Initialization is
     * bypassed.
     * 
     * @param clazz
     * @param oldValues old values of the element (or null)
     * @return
     */
    public static IElement instantiate(ElementClass clazz, Map<Key, Object> oldValues)
    {
        IElement e = new Element(clazz);
        if (oldValues!=null)
            e.setHints(oldValues);
        return e;
    }

    private ElementClass clazz;
    private IDiagram diagram;

    public Element(ElementClass clazz)
    {
        assert(clazz!=null);
        this.clazz = clazz;
    }

    @Override
    public IDiagram getDiagram() {
        assert(diagram!=null);
        return diagram;
    }

    @Override
    public IDiagram peekDiagram() {
        return diagram;
    }

    @Override
    public ElementClass getElementClass() {
        return clazz;
    }

    /**
     * This is an internal mechanism to be used only when in complete
     * comprehension of the effects. Care must be taken to install the right
     * hints when setting a new element class.
     * 
     * @param newClass
     * @param newHints
     */
    public void setElementClass(ElementClass newClass) {
        if (newClass == null)
            throw new IllegalArgumentException("null element class");
        clazz = newClass;
    }

    @Override
    public String toString() {
        String text = getHint(ElementHints.KEY_TEXT);
        if (text==null) text = "";
        Object dada = getHint(ElementHints.KEY_OBJECT);
        if (dada==null) dada = "";
        String clazzId = clazz.getId();
        String clazzStr = clazzId != null && !clazzId.isEmpty() ? clazzId
                : (clazz.getClass().getSimpleName() + "@" + clazz.hashCode());
        return clazzStr + "[" + super.toString() + " " + text + " " + dada + "]";
    }

    @Override
    public void addedToDiagram(IDiagram diagram) {
        this.diagram = diagram;
    }

    @Override
    public void destroy() {
        Element.fireDestroyed(this);
    }

    public static void fireDestroyed(IElement e)
    {
        for (LifeCycle lc : e.getElementClass().getItemsByClass(LifeCycle.class))
            lc.onElementDestroyed(e);
    }

    public static void fireCreated(IElement e)
    {
        for (LifeCycle lc : e.getElementClass().getItemsByClass(LifeCycle.class))
            lc.onElementCreated(e);
    }

    @Override
    public void dispose() {
        clearWithoutNotification();
    }

    public <T> T adapt(Class<T> adapter) {
        for (ElementAdapter ea : clazz.getItemsByClass(ElementAdapter.class))
        {
            T result = ea.adapt(this, adapter);
            if (result!=null) return result;
        }
        return null;
    }

}
