/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.diagram.handler;

import java.util.Collection;
import java.util.Objects;

import org.simantics.g2d.diagram.IDiagram;

/**
 * A diagram handler for managing and querying arbitrary relationships between
 * diagram objects. The management is done by claiming and denying relationships
 * about the objects. A single relationship can only exist once or not at all
 * between two objects, i.e. no duplicate relationships can be added.
 * 
 * <p>
 * There is no pre-defined meaning for a relationship - it is completely
 * customizable by implementing your own {@link Relationship} interface.
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public interface RelationshipHandler extends DiagramHandler {

    /**
     * A simple holder class for a single relationship. Holds all the
     * ingredients: two elements and the role of the relationship. The
     * relationship is either uni- or bi-directional.
     */
    public final class Relation {
        private final Object       subject;
        private final Relationship relationship;
        private final Object       object;

        public Relation(Object subject, Relationship relationship, Object object) {
            assert subject != null;
            assert relationship != null;
            assert object != null;
            this.subject = subject;
            this.relationship = relationship;
            this.object = object;
        }

        public Object getSubject() {
            return subject;
        }

        public Relationship getRelationship() {
            return relationship;
        }

        public Object getObject() {
            return object;
        }

        @Override
        public int hashCode() {
            return ((Objects.hashCode(subject) * 31)
                    + Objects.hashCode(object) * 31)
                    + Objects.hashCode(relationship);
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == this)
                return true;
            if (!(obj instanceof Relation))
                return false;
            Relation other = (Relation) obj;
            return subject.equals(other.subject) && object.equals(other.object) && relationship.equals(other.relationship);
        }

        @Override
        public String toString() {
            return "(" + subject + ", " + relationship + ", " + object + ")";
        }
    }

    /**
     * Claim a single relationship between the specified subject and object. An
     * inverse relation will also be claimed if one exists.
     * 
     * Subject and object should be alive (not destroyed). However it should not
     * be enforced to be activated within the specified diagram, at the moment
     * of invocation. This is because the relationship may be denied while the
     * elements are still in the process of being added to or removed from a
     * diagram.
     * 
     * @param diagram the diagram to operate on
     * @param subject the subject of the relationship
     * @param predicate the relationship itself
     * @param object the object of the relationship
     */
    void claim(IDiagram diagram, Object subject, Relationship predicate, Object object);

    /**
     * Deny a relationship. This will also deny the inverse relationship if one
     * exists.
     * 
     * Subject and object should be alive (not destroyed). However it should not
     * be enforced to be activated within the specified diagram, at the moment
     * of invocation. This is because the relationship may be denied while the
     * elements are still in the process of being added to or removed from a
     * diagram.
     */
    void deny(IDiagram diagram, Object subject, Relationship predicate, Object object);
    void deny(IDiagram diagram, Relation relation);
    void denyAll(IDiagram diagram, Object element);

    /**
     * Get all the relationships that are recorded for the specified element.
     * 
     * @param diagram the diagram to operate on
     * @param element the element to get all relations for
     * @param result a collection for gathering the result of the query or null
     *        to create a new collection for the result.
     * @return
     */
    Collection<Relation> getRelations(IDiagram diagram, Object element, Collection<Relation> result);

}
