/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.canvas.impl;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.simantics.g2d.canvas.IMouseCaptureContext;
import org.simantics.g2d.canvas.IMouseCaptureHandle;
import org.simantics.g2d.canvas.IMouseCaptureHandleListener;
import org.simantics.g2d.canvas.IMouseCaptureListener;
import org.simantics.utils.threads.IThreadWorkQueue;
import org.simantics.utils.threads.SyncListenerList;

/**
 * @author Toni Kalajainen
 */
public class MouseCaptureContext implements IMouseCaptureContext {

    protected Map<Integer, Set<IMouseCaptureHandle>> mouseCapture = 
        new HashMap<Integer, Set<IMouseCaptureHandle>>();
    
    protected SyncListenerList<IMouseCaptureListener> mouseCaptureListeners =
        new SyncListenerList<IMouseCaptureListener>(IMouseCaptureListener.class);
    
    public void addMouseCaptureListener(IMouseCaptureListener listener) {
        mouseCaptureListeners.add(listener);
    }
    public void addMouseCaptureListener(IMouseCaptureListener listener, IThreadWorkQueue thread) {
        mouseCaptureListeners.add(thread, listener);
    }    
    public void removeMouseCaptureListener(IMouseCaptureListener listener) {
        mouseCaptureListeners.remove(listener);
    }
    public void removeMouseCaptureListener(IMouseCaptureListener listener, IThreadWorkQueue thread) {
        mouseCaptureListeners.remove(thread, listener);
    }
    
    private final static Method onMouseCaptured = SyncListenerList.getMethod(IMouseCaptureListener.class, "onMouseCaptured");
    private final static Method onMouseReleased = SyncListenerList.getMethod(IMouseCaptureListener.class, "onMouseReleased");
    protected void fireMouseCaptured(int mouseId) {
        mouseCaptureListeners.fireEventSync(onMouseCaptured, this, mouseId);
    }
    protected void fireAsyncMouseCaptured(int mouseId) {
        mouseCaptureListeners.fireEventAsync(onMouseCaptured, this, mouseId);
    }
    protected void fireMouseReleased(int mouseId) {
        mouseCaptureListeners.fireEventSync(onMouseReleased, this, mouseId);
    }
    protected void fireAsyncMouseReleased(int mouseId) {
        mouseCaptureListeners.fireEventAsync(onMouseReleased, this, mouseId);
    }
    
    public synchronized IMouseCaptureHandle captureMouse(int mouseId) {
        Set<IMouseCaptureHandle> captures = mouseCapture.get(mouseId);
        IMouseCaptureHandle capture = new MouseCaptureHandle(mouseId);
        capture.addMouseCaptureHandleListener(new IMouseCaptureHandleListener() {
            @Override
            public void onCaptureReleased(IMouseCaptureHandle sender) {
                synchronized(MouseCaptureContext.this)
                {
                    Set<IMouseCaptureHandle> captures = mouseCapture.get(sender.mouseId());
                    if (!captures.contains(this)) return;
                    captures.remove(this);
                    if (captures.isEmpty()) {
                        mouseCapture.remove(captures);
                        fireMouseReleased(sender.mouseId());
                    }
                }
            }});
        if (captures==null) {
            captures = new HashSet<IMouseCaptureHandle>();
            mouseCapture.put(mouseId, captures);
            fireMouseCaptured(mouseId);
        }
        captures.add(capture);
        return capture;
    }

    

}
