/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.fastlz.impl;

import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import org.simantics.fastlz.ARCHType;
import org.simantics.fastlz.OSType;

/**
 * @author Tuukka Lehtonen
 */
public class OS {

    /**
     * Extracts the specified source file in the specified bundle into the
     * specified local directory.
     * 
     * @param url the source URL to stream the resource from
     * @param targetFile the target file to write the resource to
     * @param deleteOnExit <code>true</code> to use {@link File#deleteOnExit()}
     *        on the resulting file. Note that this does not guarantee that the
     *        file is deleted when the JVM exits
     * @return the resulting file
     * @throws FileNotFoundException
     */
    public static File copyResource(URL url, File targetFile) throws IOException, FileNotFoundException {
        FileOutputStream os = null;
        InputStream is = null;
        try {
            if (targetFile.exists())
                targetFile.delete();

            is = url.openStream();
            int read;
            byte [] buffer = new byte [16384];
            os = new FileOutputStream (targetFile);
            while ((read = is.read (buffer)) != -1) {
                os.write(buffer, 0, read);
            }
            os.close ();
            is.close ();

            return targetFile;
        } finally {
            uncheckedClose(os);
            uncheckedClose(is);
        }
    }

    public static File extractLib(URL libURL, String libName) throws FileNotFoundException, IOException {
        String tmpDirStr = System.getProperty("java.io.tmpdir");
        if (tmpDirStr == null)
            throw new NullPointerException("java.io.tmpdir property is null");
        File tmpDir = new File(tmpDirStr);
        File libFile = new File(tmpDir, libName);
        return copyResource(libURL, libFile);
    }

    public static void uncheckedClose(Closeable closeable) {
        try {
            if (closeable != null)
                closeable.close();
        } catch (IOException e) {
            //ignore
        }
    }

    public static String formOsArchSuffix() {
        String osName = System.getProperty("os.name");
        String osArch = System.getProperty("os.arch");
        OSType os = OSType.calculate();
        ARCHType arch = ARCHType.calculate();

        if (os == OSType.UNKNOWN)
            throw new UnsatisfiedLinkError("unknown OS '" + osName + "' cannot load native fastlz library");
        if (arch == ARCHType.UNKNOWN)
            throw new UnsatisfiedLinkError("unknown architecture '" + osArch + "' cannot load native fastlz library");

        String lib = "";
        switch (os) {
            case APPLE:
                lib += "-darwin";
                switch (arch) {
                    case PPC:
                        lib += "-ppc";
                        break;
                    case PPC_64:
                        lib += "-ppc_64";
                        break;
                    case X86:
                        lib += "-x86";
                        break;
                    case X86_64:
                        lib += "-x86_64";
                        break;
                    default:
                        throw new UnsatisfiedLinkError("Unsupported architecture for Apple OS: " + osArch);
                }
                break;
            case LINUX:
                lib += "-linux";
                switch (arch) {
                    case X86:
                        lib += "-x86";
                        break;
                    case X86_64:
                        lib += "-x86_64";
                        break;
                    default:
                        throw new UnsatisfiedLinkError("Unsupported architecture for Linux OS: " + osArch);
                }
                break;
            case SUN:
                lib += "-sun";
                switch (arch) {
                    case SPARC:
                        lib += "-sparc";
                        break;
                    case X86_64:
                        lib += "-x86_64";
                        break;
                    default:
                        throw new UnsatisfiedLinkError("Unsupported architecture for Sun OS: " + osArch);
                }
                break;
            case WINDOWS:
                lib += "-windows";
                switch (arch) {
                    case X86:
                        lib += "-x86";
                        break;
                    case X86_64:
                        lib += "-x86_64";
                        break;
                    default:
                        throw new UnsatisfiedLinkError("Unsupported architecture for Windows: " + osArch);
                }
                break;
            default:
                throw new UnsatisfiedLinkError("Unsupported operating system: " + os);
        }
        return lib;
    }

    public static String resolveLibName() {
        String lib = "fastlz";
        lib = System.mapLibraryName(lib + formOsArchSuffix());
        return lib;
    }

}
