/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.event;

import java.util.UUID;

import org.simantics.databoard.Bindings;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.event.ontology.EventResource;
import org.simantics.event.writer.EventSourceResolver;
import org.simantics.event.writer.EventWriteTask;
import org.simantics.event.writer.EventWriterJob;
import org.simantics.layer0.Layer0;

/**
 * @author Tuukka Lehtonen
 */
public class EventSupport {

    private Resource model;
    private Resource run;
    private EventSourceResolver.Filter filter;

    @SuppressWarnings("unused")
    private Resource eventLog;

    private EventWriterJob writer;
    private EventSourceResolver resolver;

    public EventSupport(Resource model, Resource run, EventSourceResolver.Filter filter) {
        this.model = model;
        this.run = run;
        this.filter = filter;
    }

    public Resource initialize(WriteGraph graph) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        final EventResource EVENT = EventResource.getInstance(graph);

        Resource log = graph.newResource();
        this.eventLog = log;
        graph.claim(log, L0.InstanceOf, null, EVENT.EventLog);
        graph.claimLiteral(log, L0.HasName, UUID.randomUUID().toString(), Bindings.STRING);
        graph.addLiteral(log, EVENT.HasModificationCounter, EVENT.HasModificationCounter_Inverse, L0.Integer, 0, Bindings.INTEGER);
        graph.claim(model, EVENT.HasEventLog, log);
        graph.claim(model, L0.ConsistsOf, log);

        // Switch write to graph of run resource
        final Resource _log = log;
        VirtualGraphSupport support = graph.getService(VirtualGraphSupport.class);
        graph.syncRequest(new WriteRequest( support.getGraph(graph, run) ) {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                graph.claim(_log, EVENT.HasEventProducer, run);
            }
        });

        // Start event writer & resolver services
        this.resolver = new EventSourceResolver(graph.getProvider(), log, filter);
        this.writer = new EventWriterJob(graph.getProvider(), log, this.resolver);
        this.resolver.schedule();

        return log;
    }

    public void queue(EventWriteTask task) {
        EventWriterJob w = writer;
        if (w != null)
            w.queue(task);
    }

    public void dispose() {
        if (writer != null) {
            writer.dispose();
            writer = null;
        }
        if (resolver != null) {
            resolver.dispose();
            resolver = null;
        }
    }

}
