/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.document;

import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.exception.DatabaseException;
import org.simantics.image2.ontology.ImageResource;

public class DocumentDialect {

	final static public DocumentDialect INSTANCE = new DocumentDialect();

	public static final String SIMANTICS_INTERNAL_URI_PREFIX = "http://simantics-internal/";

	private static final String HTTP = "http://";
	private static final String ROOT = "root:/";
	private static final String IMAGE = "image";
	private static final String MEDIA = "media";

	final Pattern imageOrMediaPattern = Pattern.compile("\\[\\[([Ii]mage|[Mm]edia)(?::([^\\]]+))?\\]\\]");

	private static String digest(byte[] bytes) throws NoSuchAlgorithmException {
		MessageDigest md = MessageDigest.getInstance("MD5");
		md.update(bytes);
		BigInteger number = new BigInteger(1, md.digest());
		return number.toString(16);
	}

	private static String imageExtension(ReadGraph graph, Resource image) throws DatabaseException {
		   ImageResource IMAGE = ImageResource.getInstance(graph);
		if (graph.isInstanceOf(image, IMAGE.PngImage))
			return ".png";
		else if (graph.isInstanceOf(image, IMAGE.JpegImage))
			return ".jpg";
		return null;
	}

	public String transform(ReadGraph graph, Resource res, String type, String options) {
		try {
			String[] parts = options.split("\\|");
			if (parts.length > 0) {
				String uri = parts[0];

				Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(res));
				if (indexRoot == null) return null;
				String rootURI = graph.getURI(indexRoot);

				uri = uri.replace(ROOT, rootURI);

				// (Apros #12268) For more user-friendly linking, allow users to
				// write white space as ' ' in the wiki-style image links instead
				// of having to write %20.
				uri = uri.replace(" ", "%20");

				if (IMAGE.equals(type)) {
					Resource image = graph.getPossibleResource(uri);
					if (image == null)
						return null;

					String extension = imageExtension(graph, image);
					if (extension == null)
						return null;

					byte[] bytes = graph.getValue(image, Bindings.BYTE_ARRAY);
					String digest = digest(bytes);

					Path dir = Simantics.getTemporaryDirectory("documentImages").toPath();
					Files.createDirectories(dir);

					Path f = dir.resolve(digest + extension);
					if (!Files.exists(f))
						Files.write(f, bytes);

					StringBuilder sb = new StringBuilder(128);
					sb.append("[[File: ").append(f.toUri());
					for (int i = 1; i < parts.length; i++)
						sb.append("|").append(parts[i]);
					sb.append("]]");
					return sb.toString();
				} else if (MEDIA.equals(type)) {
					Resource image = graph.getPossibleResource(uri);
					if (image == null)
						return null;

					StringBuilder sb = new StringBuilder();
					sb.append("[").append(SIMANTICS_INTERNAL_URI_PREFIX).append(uri);
					for (int i = 1; i < parts.length; i++)
						sb.append(" ").append(parts[i]);
					sb.append("]");
					return sb.toString();
				}

			}
		} catch (DatabaseException e) {
		} catch (NoSuchAlgorithmException e) {
		} catch (IOException e) {
		}
		
		return null;
	}

	public String apply(ReadGraph graph, Resource res, String markup) throws DatabaseException {

		StringBuffer sb = new StringBuffer();
		Matcher matcher = imageOrMediaPattern.matcher(markup);
		while(matcher.find()) {
			matcher.appendReplacement(sb, "");
			String type = matcher.group(1);
			String options = matcher.group(2);
			String match = transform(graph, res, type.toLowerCase(), options);
			if(match != null) sb.append(match);
			else sb.append("[[Image:" + options + "]]");
		}
		matcher.appendTail(sb);
		return sb.toString();

	}
	
}