/*******************************************************************************
 * Copyright (c) 2013, 2014 Association for Decentralized 
 * Information Management in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the THTH Simantics 
 * Division Member Component License which accompanies this 
 * distribution, and is available at
 * http://www.simantics.org/legal/sdmcl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.document.server.io;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

public class JSONObjectUtils {
    
    public static String getText(IJSONObject object) {
        return getValueOrDefault(object, "text", "");
    }
    
    public static String getLink(IJSONObject object) {
        return getValueOrDefault(object, "link", "");
    }
    
    public static String getTarget(IJSONObject object) {
        return getValueOrDefault(object, "target", "");
    }
    
    public static String getHyperLinkTarget(IJSONObject object){
    	return getValueOrDefault(object, "hyperlinkTarget", "");
    }
    
    public static String getWidth(IJSONObject object) {
        return getValueOrDefault(object, "width", "");
    }
    
    public static String getHeight(IJSONObject object) {
        return getValueOrDefault(object, "height", "");
    }

    public static String getParent(IJSONObject object) {
    	return object.getValue("parent");
    }
    
    public static boolean enabled(IJSONObject object) {
        return getValueOrDefault(object, "enabled", true);
    }
    
    public static boolean visible(IJSONObject object) {
        return getValueOrDefault(object, "visible", true);
    }
    
    public static boolean readOnly(IJSONObject object) {
        return getValueOrDefault(object, "readOnly", false);
    }
    
    public static boolean selected(IJSONObject object) {
        return getValueOrDefault(object, "selected", false);
    }

    public static boolean followEditMode(IJSONObject object) {
        return getValueOrDefault(object, "followEditMode", true);
    }

    public static String getType(IJSONObject object) {
    	String result = (String)object.getValue("type");
    	if(result == null) throw new IllegalStateException("No type for object " + object);
        return result;
    }

    public static String getId(IJSONObject object) {
    	String result = (String)object.getValue("id");
    	if(result == null) throw new IllegalStateException("No type for object " + object);
        return result;
    }
    
    public static <T> T getValueOrDefault(IJSONObject object, String key, T defaultValue) {
        try {
        	T value = (T)object.getValue(key);
        	if(value != null) return value;
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        return defaultValue;        
    }
    
    public static String[] getStringArrayValueOrDefault(IJSONObject object, String key, String[] defaultValue) {
        try {
            Object o = object.getValue(key);
            if (o instanceof List) {
                @SuppressWarnings("unchecked")
                List<String> s = (List<String>) o;
                return s.toArray(new String[s.size()]);
            } else if (o instanceof String[]) {
                return (String[]) o;
            }
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        return defaultValue;
    }

    public static float[] getFloatArrayValueOrDefault(IJSONObject object, String key, float[] defaultValue) {
    	try {
        	Object o = object.getValue(key);
            if (o instanceof List) {
                @SuppressWarnings("unchecked")
                List<Float> list = (List<Float>) o;
                float[] result = new float[list.size()];
                int pos = 0;
                for (float value : list) {
                	result[pos++] = value;
                }
                return result;
            } else if (o instanceof float[]) {
                return (float[]) o;
            }
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        return defaultValue;
    }

    public static int[] getIntArrayValueOrDefault(IJSONObject object, String key, int[] defaultValue) {
        try {
            Object o = object.getValue(key);
            if (o instanceof List) {
                @SuppressWarnings("unchecked")
                List<Integer> list = (List<Integer>) o;
                int[] result = new int[list.size()];
                int pos = 0;
                for (int i : list) {
                	result[pos++] = i;
                }
                return result;
            } else if (o instanceof int[]) {
                return (int[]) o;
            }
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        return defaultValue;
    }

    public static double[] getDoubleArrayValueOrDefault(IJSONObject object, String key, double[] defaultValue) {
        try {
        	Object o = object.getValue(key);
            if (o instanceof List) {
                @SuppressWarnings("unchecked")
                List<Double> list = (List<Double>) o;
                double[] result = new double[list.size()];
                int pos = 0;
                for (double value : list) {
                	result[pos++] = value;
                }
                return result;
            } else if (o instanceof double[]) {
                return (double[]) o;
            }
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        return defaultValue;
    }
    
//    public static String[] getObjectArrayValue(IJSONObject object, String key) {
//        try {
//            String value = (String)object.getValue(key);
//            if(value != null && value.startsWith("[") && value.endsWith("]")) {
//            	
//            	if(value.length() == 2) return new String[0];
//            	
//                value = value.substring(1, value.length() - 1);
//                String[] split = value.split(",");
//                
//                for(int i = 0; i < split.length; i++) {
//                	if(split[i].startsWith("\"") && split[i].endsWith("\""))
//                			split[i] = split[i].substring(1, split[i].length() - 1);
//                }
//                
//                return split;
//            }
//        } catch (ClassCastException e) {
//            e.printStackTrace();
//            return null;
//        }
//        
//        return null;
//    }

    public static Collection<ICommand> getCommands(IJSONObject object) {
        try {
        	Collection<ICommand> result = object.getValue("commands");
        	if(result != null) return result;
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        return Collections.emptyList();
    }
    
    @SuppressWarnings("unchecked")
    public static Collection<IChartItem> getChartItems(IJSONObject object) {
        try {
            Object values = object.getValue("values");
            if (values == null)
                return Collections.emptyList();
            if (values instanceof String) {
                String valuess = (String) values;
                if (valuess.length() == 0)
                    return Collections.emptyList();
            }
            return (List<IChartItem>) values;
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        
        return Collections.emptyList();
    }
    
//    public static TaskSeriesCollection getGanttTasks(IJSONObject object) {
//        TaskSeriesCollection collection = new TaskSeriesCollection();
//        try {
//            
//            //JSONArray seriesArray = object.getJSONArray("ganttSeries");
//            @SuppressWarnings("unchecked")
//            List<IJSONObject> seriesArray = (List<IJSONObject>) object.getValue("ganttSeries");
//            for(IJSONObject seriesObject : seriesArray) {
//                //IJSONObject seriesObject = seriesArray.getJSONObject(i);
//                
//                TaskSeries series = new TaskSeries(getStringValueOrNull(seriesObject, "name"));
//                
//                //JSONArray tasksArray = seriesObject.getJSONArray("tasks");
//                @SuppressWarnings("unchecked")
//                List<IJSONObject> tasksArray = (List<IJSONObject>) object.getValue("tasks");
//                
//                for(IJSONObject taskObject : tasksArray) {
//                    //IJSONObject taskObject = tasksArray.getJSONObject(j);
//                    String name = getStringValueOrNull(taskObject, "name");
//                    Double start = getDoubleValueOrNull(taskObject, "start");
//                    Double end = getDoubleValueOrNull(taskObject, "end");
//                    
////                    System.out.println("Task: " + name + ", start: " + start + ", end: " + end);
//                    
//                    if(start >= 0 && end >= 0 && start < end) {
//                        Task task = new Task(name,  new Date(start.intValue()), new Date(end.intValue()));
//                        
//                        try {
//                            //JSONArray subtasksArray = taskObject.getJSONArray("subtasks");
//                            @SuppressWarnings("unchecked")
//                            List<IJSONObject> subtasksArray = (List<IJSONObject>) object.getValue("subtasks");
////                            System.out.println("\tFound " + subtasksArray.length() + " subtasks");
//                            for(IJSONObject subtaskObject : subtasksArray) {
//                                //IJSONObject subtaskObject = subtasksArray.getJSONObject(s);
//                                String subtaskname = getStringValueOrNull(subtaskObject, "name");
//                                Double subtaskstart = getDoubleValueOrNull(subtaskObject, "start");
//                                Double subtaskend = getDoubleValueOrNull(subtaskObject, "end");
//                                if(subtaskstart >= 0 && subtaskend >= 0 && subtaskstart < subtaskend) {
//                                    Task subtask = new Task(subtaskname + " subtask " + (subtasksArray.indexOf(subtaskObject)),  new Date(subtaskstart.intValue()), new Date(subtaskend.intValue()));
//                                    task.addSubtask(subtask);
////                                    System.out.println("\tSubTask: " + subtaskname + ", start: " + subtaskstart + ", end: " + subtaskend);
//                                }
//                            }
//                        } catch (ClassCastException e) {
//                            // Some tasks may not have subtasks
//                            e.printStackTrace();
//                        }
//                        
//                        series.add(task);
//                    }
//                }
//                
//                collection.add(series);
//            }
//        } catch (ClassCastException e) {
//            e.printStackTrace();
//        }
//        return collection;
//    }
    
    @SuppressWarnings("unchecked")
    public static Collection<IDataDefinition> getDataDefinitions(IJSONObject object) {
        try {
        	Collection<IDataDefinition> result = (List<IDataDefinition>) object.getValue("dataDefinitions");
        	if(result != null) return result;
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        return Collections.emptyList();
    }
    
    @SuppressWarnings("unchecked")
    public static Collection<ITableCell> getTableCell(IJSONObject object) {
        try {
            Object tableCells = object.getValue("tableCells");
            if (tableCells instanceof String) {
                String tableCellsS = (String) tableCells;
                if (tableCellsS.length() == 0)
                    return Collections.emptyList();
            }
            if (tableCells != null) {
            	return (List<ITableCell>) tableCells;
            } else {
            	return Collections.emptyList();
            }
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        return Collections.emptyList();
    }
    
    @SuppressWarnings("unchecked")
    public static Collection<ITreeTableCell> getTreeTableCells(IJSONObject object) {
        try {
            Object tableCells = object.getValue("tableCells");
            if (tableCells instanceof String) {
                String tableCellsS = (String) tableCells;
                if (tableCellsS.length() == 0)
                    return Collections.emptyList();
            }
            if (tableCells != null) {
            	return (List<ITreeTableCell>) tableCells;
            } else {
            	return Collections.emptyList();
            }
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        return Collections.emptyList();
    }

    public static Collection<FileInfo> getFiles(IJSONObject object) {
        try {
            @SuppressWarnings("unchecked")
            List<IJSONObject> fileArray = (List<IJSONObject>) object.getValue("files");
            ArrayList<FileInfo> fileList = new ArrayList<FileInfo>();
            for(IJSONObject f : fileArray) {
                String name = getValueOrDefault(f, "name", "");
                String content = getValueOrDefault(f, "content", "");
                Long resourceId = getValueOrDefault(f, "resourceId", 0L);
                Long creationTimeMillis = getValueOrDefault(f, "creationTimeMillis", 0L);
                boolean canModify = getValueOrDefault(f, "canModify", false);
                if(name != null && content != null) {
                    fileList.add(new FileInfo(name, content, resourceId, creationTimeMillis, canModify));
                }
            }
            return fileList;
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        
        return Collections.emptyList();
    }

    public static List<IListItem> getListItems(IJSONObject object) {
        Object items =  object.getValue("listItems");
        if (items instanceof List<?>)
            return (List<IListItem>) items;
        else
            return Collections.<IListItem>emptyList();
    }
    
    @SuppressWarnings("unchecked")
    public static Collection<ITreeTableCell> getTreeTableCell(IJSONObject object) {
        try {
            Object treeTableCells = object.getValue("tableCells");
            if (treeTableCells instanceof String) {
                String tableCellsS = (String) treeTableCells;
                if (tableCellsS.length() == 0)
                    return Collections.emptyList();
            }
            if (treeTableCells != null) {
                return (List<ITreeTableCell>) treeTableCells;
            } else {
                return Collections.emptyList();
            }
        } catch (ClassCastException e) {
            e.printStackTrace();
        }
        return Collections.emptyList();
    }

    public static final boolean equalObjects(Object oldValue, Object newValue) {
    	if (newValue != null) {
    		if (newValue.getClass().isArray()) {
    			return arrayEquals(newValue, oldValue);
    		} else {
    			return newValue.equals(oldValue);
    		}
    	} else
    		return oldValue == null;
    }

    
	/**
	 * @param av1 an array (guaranteed)
	 * @param av2 any object
	 * @return <code>true</code> if the two arrays are equal
	 */
	private static final boolean arrayEquals(Object av1, Object av2) {
		if (av2 == null)
			return false;
		Class<?> c1 = av1.getClass().getComponentType();
		Class<?> c2 = av2.getClass().getComponentType();
		if (c2 == null || !c1.equals(c2))
			return false;
		boolean p1 = c1.isPrimitive();
		boolean p2 = c2.isPrimitive();
		if (p1 != p2)
			return false;
		if (!p1)
			return Arrays.equals((Object[]) av1, (Object[]) av2);
		if (boolean.class.equals(c1))
			return Arrays.equals((boolean[]) av1, (boolean[]) av2);
		else if (byte.class.equals(c1))
			return Arrays.equals((byte[]) av1, (byte[]) av2);
		else if (int.class.equals(c1))
			return Arrays.equals((int[]) av1, (int[]) av2);
		else if (long.class.equals(c1))
			return Arrays.equals((long[]) av1, (long[]) av2);
		else if (float.class.equals(c1))
			return Arrays.equals((float[]) av1, (float[]) av2);
		else if (double.class.equals(c1))
			return Arrays.equals((double[]) av1, (double[]) av2);
		throw new RuntimeException("Unsupported objects for equality testing ." + av1 + " vs. " + av2);
		
	}
    
    
}
