package org.simantics.document.linking.wizard;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.document.linking.report.templates.CompleteStructureWriter;
import org.simantics.document.linking.report.templates.DiagramStructureWithDependenciesWriter;
import org.simantics.document.linking.report.templates.DiagramStructureWriter;
import org.simantics.document.linking.report.templates.DocumentStructureWriter;
import org.simantics.document.linking.report.templates.ModelDocumentWriter;
import org.simantics.document.linking.report.templates.ReferredDocumentWriter;
import org.simantics.document.linking.report.templates.ReportWriter;
import org.simantics.document.linking.views.ModelRead;
import org.simantics.document.linking.views.ModelToComboListener;
import org.simantics.ui.SimanticsUI;


/**
 * Wizard page for selecting report type and file.
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class ReportSelectionPage extends WizardPage{
	
	protected ReportSelectionPage(String pageName) {
		super(pageName,pageName,null);
		setPageComplete(false);
	}

	CCombo modelCombo;
	Text filenameText;
	Button browseButton;
	
	ModelToComboListener modelToComboListener;
	
	List<Button> reportSelectionButtons = new ArrayList<Button>();
	List<ReportWriter<?>> reportWriters = new ArrayList<ReportWriter<?>>();
	
	ReportWriter<?> selectedWriter;
	Resource selectedModel;
	Resource input;
	
	@Override
	public void createControl(Composite parent) {
		
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout(3,false));
		Label label = new Label(composite, SWT.NONE);
		label.setText(Messages.ReportSelectionPage_Model);
		modelCombo = new CCombo(composite, SWT.BORDER|SWT.READ_ONLY);
		
		label = new Label(composite, SWT.NONE);
		label.setText(Messages.ReportSelectionPage_File);
		filenameText = new Text(composite, SWT.BORDER|SWT.SINGLE);
		browseButton = new Button(composite, SWT.PUSH);
		browseButton.setText(Messages.ReportSelectionPage_Browse);
		
		reportWriters.add(new ModelDocumentWriter());
		reportWriters.add(new ReferredDocumentWriter());
		reportWriters.add(new DocumentStructureWriter());
		reportWriters.add(new DiagramStructureWriter());
		reportWriters.add(new DiagramStructureWithDependenciesWriter());
		reportWriters.add(new CompleteStructureWriter());
		
		Group group = new Group(composite, SWT.NONE);
		group.setText(Messages.ReportSelectionPage_ReportTemplates);
		group.setLayout(new FillLayout(SWT.VERTICAL));
		
		GridDataFactory.fillDefaults().grab(true, false).align(SWT.FILL, SWT.CENTER).applyTo(filenameText);
		GridDataFactory.fillDefaults().span(2, 1).applyTo(modelCombo);
		GridDataFactory.fillDefaults().span(3, 1).applyTo(group);
		
		browseButton.addSelectionListener(new SelectionAdapter() {
			
			@Override
			public void widgetSelected(SelectionEvent e) {
				FileDialog dialog = new FileDialog(Display.getCurrent().getActiveShell(),SWT.SAVE);
				dialog.setFilterExtensions(new String[]{"*.pdf","*.html"}); //$NON-NLS-1$ //$NON-NLS-2$
				dialog.setFilterNames(new String[]{Messages.ReportSelectionPage_FilterPDFDocument,Messages.ReportSelectionPage_FilterHTMLDocument});
				String filename = dialog.open();
				if (filename == null)
					filenameText.setText(""); //$NON-NLS-1$
				else
					filenameText.setText(filename);
				validate();
			}
		});
		
		
		SelectionListener listener = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button selected = (Button)e.widget;
				select(selected);
			}
		};
		
		for (ReportWriter<?> rw : reportWriters) {
			Button b = new Button(group, SWT.RADIO);
			b.setText(rw.getName());
			b.setData(rw);
			b.addSelectionListener(listener);
			reportSelectionButtons.add(b);
		}
		
		Simantics.getSession().asyncRequest(new ReadRequest() {
			
			@Override
			public void run(ReadGraph graph) throws DatabaseException {
				Resource project = Simantics.getProject().get();
				modelToComboListener = new ModelToComboListener(modelCombo) {
					@Override
					public Resource getCurrentModel() {
						return selectedModel;
					} 
					
					@Override
					public void loaded() {
						preselectModel();
					}
				};
				graph.syncRequest(new ModelRead(project),modelToComboListener);
				
			}
		});
		
		modelCombo.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				int index = modelCombo.getSelectionIndex();
				if (index == -1)
					return;
				Resource model = (Resource)modelCombo.getData(Integer.toString(index));
				if (model != null)
					selectedModel = model;
				validate();
			}
		});
		
		select(reportSelectionButtons.get(0));
		setControl(composite);
	}
	
	public List<Button> getReportSelectionButtons() {
		return reportSelectionButtons;
	}
	
	private void select(Button selected) {
		selected.setSelection(true);
		for (Button b : reportSelectionButtons) {
			if (b != selected)
				b.setSelection(false);
		}
		selectedWriter = (ReportWriter<?>)selected.getData();
		validate();
	}
	
	private void validate() {
		setPageComplete(selectedWriter!= null && selectedModel != null && filenameText.getText().length() > 0);
	}
	
	public ReportWriter<?> getSelectedWriter() {
		return selectedWriter;
	}
	
	public String getFilename() {
		return filenameText.getText();
	}
	
	public Resource getSelectedModel() {
		return selectedModel;
	}
	
	public void setInput(Resource input) {
		this.input = input;
	}
	
	private void preselectModel() {
		if (input != null) {
			for (int i = 0; i < modelCombo.getItemCount(); i++) {
				if (input.equals(modelCombo.getData(Integer.toString(i)))) {
					modelCombo.select(i);
					selectedModel = input;
				}
			}
		}
	}
	
	@Override
	public void setVisible(boolean visible) {
		if (visible) {
			preselectModel();
		}
		super.setVisible(visible);
	}

}
