package org.simantics.document.linking.report.html;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URL;
import java.net.URLDecoder;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.osgi.framework.Bundle;
import org.simantics.document.linking.Activator;
import org.simantics.document.linking.report.Document;
import org.simantics.document.linking.report.DocumentElement;
import org.simantics.document.linking.report.DocumentItem;
import org.simantics.document.linking.report.DocumentTitlePage;
import org.simantics.document.linking.report.Table;
import org.simantics.document.linking.report.TableOfContents;
import org.simantics.document.linking.report.TextItem;
import org.simantics.document.linking.report.URLItem;
import org.simantics.document.linking.report.base.TextItemImpl;
import org.simantics.utils.datastructures.Arrays;


/**
 * HTML format reporter.
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class HTMLDocument extends HTMLStreamElement implements Document {
	
	
	boolean referCSS = false;  // if true, generated report refers to bas css in the plug-in. Otherwise the base css's contents are copied to report file.
	
	int currentPage = 0;
	int currentLine = 1;

	HTMLStreamElement content;
	
	private Map<Class<? extends HTMLElement>, Integer> uniqueIndex = new HashMap<Class<? extends HTMLElement>, Integer>();
	
	
	@Override
	public String getId() {
		return null;
	}
	
	public HTMLDocument(File file) throws Exception {
		super(file);
		content = new HTMLStreamElement(this);
	}
	
	String getUniqueId(Class<? extends HTMLElement> cls) {
		Integer index = uniqueIndex.get(cls);
		if (index == null)
			index = 1;
		else
			index = index + 1;
		uniqueIndex.put(cls,index);
		
		return cls.getSimpleName() +"_"+index; //$NON-NLS-1$
	}
	String getUniqueId(HTMLElement element) {
		Class<? extends HTMLElement> cls = element.getClass();
		return getUniqueId(cls);
	}
	
	private static String getDataUrl() throws IOException {
		 Bundle b = Platform.getBundle(Activator.PLUGIN_ID);
		 URL dataUrl = FileLocator.find(b, new Path("report"), null); //$NON-NLS-1$
		 URL fileUrl = FileLocator.toFileURL(dataUrl);
		 return URLDecoder.decode(fileUrl.getPath(), "UTF-8");  //$NON-NLS-1$
	}
	
	private void copyBaseCSS(String cssUrl) throws Exception {
		File f = new File(URLDecoder.decode(cssUrl, "UTF-8")).getAbsoluteFile(); //$NON-NLS-1$
		copyData(f, os);
	}

	
	private void header() throws Exception {
		PrintStream resultOs = os;
		String cssUrl = getDataUrl() +"/report.css"; //$NON-NLS-1$
		resultOs.println("<!DOCTYPE html>"); //$NON-NLS-1$
		resultOs.println("<html lang=\"en\">"); //$NON-NLS-1$
		resultOs.println("<head>"); //$NON-NLS-1$
		resultOs.println("  <meta charset=\"utf-8\">"); //$NON-NLS-1$
		resultOs.println("  <title>Report</title>"); //$NON-NLS-1$
		if (referCSS)
			resultOs.println("  <link rel=\"stylesheet\" href=\"" + cssUrl + "\" type=\"text/css\">"); //$NON-NLS-1$ //$NON-NLS-2$
		resultOs.println("  <style>"); //$NON-NLS-1$
		if (!referCSS) {
			copyBaseCSS(cssUrl);
		}
		
		resultOs.println("  </style>"); //$NON-NLS-1$
		resultOs.println("</head>"); //$NON-NLS-1$
		resultOs.println("<body>"); //$NON-NLS-1$
	}

	private void footer() throws Exception{
		if (currentTable != null)
			currentTable.endTable();
		if (toc != null)
			toc.close();
		content.close();
		PrintStream resultOs = os;
		resultOs.println("</body>"); //$NON-NLS-1$
		resultOs.println("</html>"); //$NON-NLS-1$
	}
	
	@Override
	public void close() throws Exception {
		footer();
		super.close();
	}
	

	@Override
	public int getCurrentLine() {
		return currentLine;
	}
	
	@Override
	public int getAvailableLines() {
		return Integer.MAX_VALUE;
	}
	
	@Override
	public void nextPage() throws Exception {
		if (currentPage == 0)
			header();	
		currentPage++;
		currentLine = 1;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <T extends DocumentElement> T newElement(Class<T> cls,String...options) throws Exception {
		if (cls == Table.class) {
			return (T)newTable(Arrays.contains(options, Document.TOC));
		} else if (cls == TableOfContents.class) {
			return (T)getOrCreateToc();
		} else if (cls == DocumentTitlePage.class) {
			if (titlePage != null)
				throw new Exception("Document many have only one title page"); //$NON-NLS-1$
			titlePage = new HTMLTitlePage(this);
			return (T)titlePage;
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <T extends DocumentElement> T nextElement(Class<T> cls,String...options)	throws Exception {
		if (cls == Table.class) {
			return (T)nextTable(Arrays.contains(options, Document.TOC));
		} else if (cls == TableOfContents.class) {
			return (T)getOrCreateToc();
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <T extends DocumentElement> T getCurrentElement(Class<T> cls) {
		if (cls == Table.class) {
			return (T)getCurrentTable();
		} else if (cls == TableOfContents.class) {
			return (T)toc;
		} else if (cls == DocumentTitlePage.class) {
			return (T)titlePage;
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <T extends DocumentItem> T newItem(Class<T> cls, String... options)
			throws Exception {
		if (cls == TextItem.class)
			return (T)new TextItemImpl();
		if (cls == URLItem.class)
			return (T)new HTMLURLItemImpl();
		return null;
	}
	
	private HTMLTable currentTable;
	
	
	public Table nextTable(boolean id) throws Exception {
		if (currentTable != null) {
			currentTable.endTable();
			currentTable = new HTMLTable(currentTable,id);
			return currentTable;
		} else {
			return currentTable;
		}
		
	}
	
	
	public Table newTable(boolean id) throws Exception {
		if (currentTable != null) {
			currentTable.endTable();
		}
		currentTable = new HTMLTable(this,content.os,id);
		return currentTable;
	}
	
	
	public Table getCurrentTable() {
		return currentTable;
	}
	
	HTMLTocElement toc;
	
	public TableOfContents getOrCreateToc() throws Exception{
		if (toc == null) {
			toc = new HTMLTocElement(this);
		}
		return toc;
	}
	
	HTMLTitlePage titlePage;

}
