/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.synchronization.graph;

import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Write;
import org.simantics.diagram.internal.DebugPolicy;
import org.simantics.diagram.internal.timing.GTask;
import org.simantics.diagram.internal.timing.Timing;
import org.simantics.diagram.synchronization.ErrorHandler;
import org.simantics.diagram.synchronization.IModification;

/**
 * A graph database write request that composes several {@link IModification}
 * instances into a single {@link Write} request.
 * 
 * @author Tuukka Lehtonen
 * 
 * @see IModification
 */
public class ModificationProcessRequest extends WriteRequest {

    ErrorHandler    errorHandler;
    IModification[] modifications;

    public ModificationProcessRequest(ErrorHandler errorHandler, IModification[] modifications) {
        this.errorHandler = errorHandler;
        this.modifications = modifications;
    }
    
    public ModificationProcessRequest(ErrorHandler errorHandler, IModification[] modifications, VirtualGraph provider) {
    	super(provider);
        this.errorHandler = errorHandler;
        this.modifications = modifications;
    }


    @Override
    public void perform(final WriteGraph g) throws DatabaseException {
        g.markUndoPoint();
        if (DebugPolicy.DEBUG_GRAPH_WRITEBACK)
            System.out.println(Thread.currentThread() + " Performing " + modifications.length + " modifications");
        Timing.timed("" + modifications.length + " MODIFICATIONS", new GTask() {
            @Override
            public void run() throws DatabaseException {
                for (IModification m : modifications) {
                    if (m != null) {
                        modification(g, m);
                    }
                }
            }
        });
    }

    public void modification(final WriteGraph g, final IModification m) throws DatabaseException {
        Timing.timed(m.toString(), new GTask() {
            @Override
            public void run() throws DatabaseException {
                synchronized (m) {
                    try {
                        if (DebugPolicy.DEBUG_GRAPH_WRITEBACK_MODIFICATION)
                            System.out.println(Thread.currentThread() + "  Performing modification: " + m);
                        m.perform(g);
                    } catch (Exception e) {
                        m.setException(e);
                        errorHandler.error(e.getMessage(), e);
                    } catch (Error e) {
                        m.setException(e);
                        errorHandler.error(e.getMessage(), e);
                    } finally {
                        m.markComplete();
                        m.notifyAll();
                    }
                }
            }
        });
    }

}