/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.handler;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ContributionItem;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.ui.contribution.DynamicMenuContribution;
import org.simantics.utils.ui.AdaptionUtils;
import org.simantics.utils.ui.ExceptionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class TerminalConnectionDirectionsMenuContribution extends DynamicMenuContribution {

	private static final IContributionItem[] NONE = {};

	private static final int[] DIRECTION_MASK = { 8, 4, 1, 2 };
	private static final String[] DIRECTION_NAME = { "Up", "Left", "Right", "Down" };

	@Override
	protected IContributionItem[] getContributionItems(ReadGraph graph, Object[] selection) throws DatabaseException {
		return getTerminalDirectionItems(graph, selection).toArray(NONE);
	}

	public static Collection<IContributionItem> getTerminalDirectionItems(ReadGraph graph, Object[] selection) throws DatabaseException {
		DiagramResource DIA = DiagramResource.getInstance(graph);

		final List<Resource> terminals = new ArrayList<Resource>();
		for (Object s : selection) {
			Resource connection = AdaptionUtils.adaptToSingle(s, Resource.class);
			if (connection == null || !graph.isInstanceOf(connection, DIA.Terminal))
				return Collections.emptyList();
			terminals.add(connection);
		}
		if (terminals.isEmpty())
			return Collections.emptyList();

		int jointMask = 0;
		for (Resource terminal : terminals) {
			Integer dirs = graph.getPossibleRelatedValue(terminal, DIA.Terminal_AllowedDirections, Bindings.INTEGER);
			int mask = dirs == null ? 0 : dirs;
			jointMask |= mask;
		}

		final List<ToggleDirectionAction> actions = new ArrayList<ToggleDirectionAction>(4);
		for (int i = 0; i < DIRECTION_NAME.length; ++i) {
			String name = DIRECTION_NAME[i];
			int mask = DIRECTION_MASK[i];
			boolean wasChecked = (jointMask & mask) != 0;
			int orMask = wasChecked ? 0 : mask;
			int andMask = wasChecked ? ~mask : 0xffffffff;
			ToggleDirectionAction action = new ToggleDirectionAction(graph.getSession(), terminals, orMask, andMask, name);
			action.setChecked(wasChecked);
			actions.add(action);
		}

		return Collections.<IContributionItem>singleton(
				new ContributionItem() {
					@Override
					public void fill(Menu menu, int index) {
						MenuItem directions = new MenuItem(menu, SWT.CASCADE, index);
						directions.setText("Allowed Routing Directions");

						Menu subMenu = new Menu(menu);
						directions.setMenu(subMenu);

						for (ToggleDirectionAction a : actions) {
							MenuItem item = new MenuItem(subMenu, SWT.CHECK);
							item.setText(a.getText());
							item.addSelectionListener(a);
							item.setSelection(a.isChecked());
						}
					}
				}
				);
	}

	static class ToggleDirectionAction extends Action implements SelectionListener {

		Session session;
		Collection<Resource> terminals;
		int orMask;
		int andMask;

		public ToggleDirectionAction(Session session,
				List<Resource> terminals, int orMask, int andMask,
				String name) {
			super(name, Action.AS_CHECK_BOX);
			this.session = session;
			this.terminals = terminals;
			this.orMask = orMask;
			this.andMask = andMask;
		}

		@Override
		public void run() {
			session.asyncRequest(new WriteRequest() {
				@Override
				public void perform(WriteGraph graph) throws DatabaseException {
					DiagramResource DIA = DiagramResource.getInstance(graph);
					for (Resource terminal : terminals) {
						Integer dirs = graph.getPossibleRelatedValue(terminal, DIA.Terminal_AllowedDirections, Bindings.INTEGER);
						int mask = dirs == null ? 0 : dirs;
						mask &= andMask;
						mask |= orMask;
						graph.claimLiteral(terminal, DIA.Terminal_AllowedDirections, mask, Bindings.INTEGER);
					}
				}
			}, parameter -> {
				if (parameter != null)
					ExceptionUtils.logError(parameter);
			});
		}

		@Override
		public void widgetDefaultSelected(SelectionEvent e) {
			widgetSelected(e);
		}

		@Override
		public void widgetSelected(SelectionEvent e) {
			run();
		}

	}

}
