/*******************************************************************************
 * Copyright (c) 2007, 2017 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.elements;

import java.awt.Graphics2D;
import java.util.List;
import java.util.Set;

import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.participant.Selection;
import org.simantics.g2d.element.IElement;
import org.simantics.scenegraph.ExportableWidget.RasterOutputWidget;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.g2d.events.EventTypes;
import org.simantics.scenegraph.g2d.events.KeyEvent.KeyPressedEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseClickEvent;
import org.simantics.scenegraph.g2d.events.command.CommandEvent;
import org.simantics.scenegraph.g2d.nodes.SVGNodeAssignment;
import org.simantics.scenegraph.g2d.nodes.SingleElementNode;
import org.simantics.scenegraph.utils.NodeUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kitfox.svg.SVGDiagram;
import com.kitfox.svg.SVGException;
import com.kitfox.svg.Text;
import com.kitfox.svg.Tspan;

@RasterOutputWidget
public class SVGNode extends org.simantics.scenegraph.g2d.nodes.SVGNode {

    private static final Logger LOGGER = LoggerFactory.getLogger(SVGNode.class);

    private static final long serialVersionUID = 4735066193941274186L;

    transient EditorStateManager esm  = null;
    transient boolean parametersProcessed = false;

    @Override
    public void cleanup() {
        if (esm != null)
            removeEventHandler(this);
        super.cleanup();
    }

    @Override
    public void render(Graphics2D g2d) {
        if (!parametersProcessed) {
            SingleElementNode sne = getSingleElementNode();
            if (sne != null) {
                Boolean enableEditing = sne.getParameter("enableEditing");
                if (enableEditing != null && enableEditing) {
                    esm = new EditorStateManager(this);
                    addEventHandler(this);
                }
                parametersProcessed = true;
            }
        }

        super.render(g2d);
    }

    @Override
    protected int dynamicHash() {
        return esm != null ? esm.currentHash() : 0;
    }

    @Override
    protected boolean applyAssignments(SVGDiagram diagram, List<SVGNodeAssignment> assignments) throws SVGException {
        boolean changed = super.applyAssignments(diagram, assignments);
        if (esm != null && !esm.applyEditMode(diagram))
            diagram.updateTime(0);
        return changed;
    }

    private boolean isSelected(IG2DNode node) {
        ICanvasContext ctx = DiagramNodeUtil.getPossibleCanvasContext(this);
        if (ctx == null)
            return false;
        IElement ie = DiagramNodeUtil.getElement(ctx, node);
        if (ie == null)
            return false;
        Selection sel = ctx.getAtMostOneItemOfClass(Selection.class);
        if (sel != null) {
            Set<IElement> elems = sel.getSelection(0);
            if (elems.size() == 1 && elems.contains(ie))
                return true;
        }
        return false;
    }

    private boolean isSelected() {
        SingleElementNode sne = getSingleElementNode();
        if (isSelected(sne))
            return true;
        INode n = sne.getParent();
        while (n != null) {
            if (isSelected((IG2DNode)n))
                return true;
            n = n.getParent();
        }
        return false;
    }

    @Override
    protected boolean mouseClicked(MouseClickEvent e) {
        if(esm != null) {
            if(esm.isEditMode()) {
                esm.applyEdit();
                esm.deactivateEdit();
                return true;
            }
            if(isSelected()) {
                if(esm.tryToStartEditMode(diagramCache, e))
                    return true;
            }
        }
        return super.mouseClicked(e);
    }

    SingleElementNode getSingleElementNode() {
        return (SingleElementNode)NodeUtil.getPossibleNearestParentOfType(this, SingleElementNode.class);
    }

    @Override
    protected boolean keyPressed(KeyPressedEvent e) {
        if (esm != null && esm.keyPressed(e))
            return true;
        return super.keyPressed(e);
    }

    @Override
    protected boolean handleCommand(CommandEvent e) {
        if (esm != null
                && !esm.isEditMode()
                && "org.eclipse.ui.edit.rename".equals(e.command.id)
                && isSelected()
                && esm.tryToStartEditMode(diagramCache)) {
            return true;
        }
        return super.handleCommand(e);
    }

    @Override
    public int getEventMask() {
        return EventTypes.KeyPressedMask | EventTypes.MouseClickMask | EventTypes.CommandMask;
    }

    public void modifyTextElement(String id, String newText) {
        Text text = (Text) diagramCache.getElement(id);

        EditorState es = new EditorState();
        es.base = new EditorStateStatic();
        es.base.textElementId = text.getId();

        Tspan span = (Tspan)text.getContent().get(0);

        try {
            span.setText(newText);
            text.rebuild();
            diagramCache.updateTime(0);
        } catch (SVGException e) {
            LOGGER.error("Failed to update text span element " + id + " with text " + newText, e);
        }
    }

}
