/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import org.simantics.databoard.Bindings;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.flag.DiagramFlagPreferences;
import org.simantics.diagram.flag.FlagLabelingScheme;
import org.simantics.diagram.flag.FlagUtil;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.diagram.synchronization.graph.ElementWriter;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.handler.DataElementMap;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.elementclass.FlagClass;
import org.simantics.g2d.elementclass.FlagHandler;
import org.simantics.layer0.Layer0;

/**
 * @author Tuukka Lehtonen
 */
public class FlagWriter implements ElementWriter {

    @Override
    public void addToGraph(WriteGraph g, IElement flag, Resource flagResource) throws DatabaseException {
        DiagramResource dia = DiagramResource.getInstance(g);
        Layer0 l0 = Layer0.getInstance(g);

        FlagHandler fh = flag.getElementClass().getSingleItem(FlagHandler.class);
        FlagClass.Type type = fh.getType(flag);
        String text = ElementUtils.getText(flag);
        if (text != null && text.isEmpty())
            text = null;

        g.deny(flagResource, dia.HasFlagType);
        g.claim(flagResource, dia.HasFlagType, null, DiagramGraphUtil.toFlagTypeResource(dia, type));

        Resource counterpart = null;

        FlagClass.Connection<IElement> conn = fh.getConnection(flag);
        if (conn != null) {
            IElement counterpartE = fh.getCorrespondence(flag);
            IDiagram od = ElementUtils.getDiagram(counterpartE);
            DataElementMap dem = od.getDiagramClass().getSingleItem(DataElementMap.class);
            counterpart = (Resource) dem.getData(od, counterpartE);
        }

        if (counterpart != null) {
            FlagUtil.join(g, flagResource, counterpart);

            Resource diagram = OrderedSetUtils.getSingleOwnerList(g, flagResource, dia.Diagram);
            Resource counterpartDiagram = OrderedSetUtils.getSingleOwnerList(g, counterpart, dia.Diagram);

            // Only give labels if flags are locally connected.
            if (diagram.equals(counterpartDiagram)) {
                // Only set labels if this is a flag pair.
                if (text == null) {
                    // Try to generate label if not specfied externally.
                    FlagLabelingScheme scheme = DiagramFlagPreferences.getActiveFlagLabelingScheme(g);
                    if (scheme != null) {
                        text = scheme.generateLabel(g, diagram);
                    }
                }

                if (text != null) {
                    g.claimLiteral(flagResource, l0.HasLabel, dia.FlagLabel, text, Bindings.STRING);
                    g.claimLiteral(counterpart, l0.HasLabel, dia.FlagLabel, text, Bindings.STRING);
                }
            }
        }
    }

    @Override
    public void removeFromGraph(WriteGraph graph, Resource flag) throws DatabaseException {
        FlagUtil.disconnectFlag(graph, flag);
    }

}
