/*******************************************************************************
 * Copyright (c) 2016 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.function;

import java.util.Objects;
import java.util.function.Consumer;

import org.simantics.db.exception.DatabaseException;

/**
 * Represents a database operation that accepts a single input argument and
 * returns no result and can throw database exceptions. Unlike most other
 * functional interfaces, {@code DbConsumer} is expected to operate via
 * side-effects.
 *
 * <p>
 * This is a functional interface whose functional method is
 * {@link #accept(Object)}. The only difference to {@link Consumer} is that this
 * allows throwing DatabaseException.
 *
 * @param <T>
 *            the type of the input to the operation
 *
 * @author Tuukka Lehtonen
 * @since 1.22
 */
@FunctionalInterface
public interface DbConsumer<T> {

    /**
     * Performs this operation on the given argument.
     *
     * @param t the input argument
     */
    void accept(T t) throws DatabaseException;

    /**
     * Returns a composed {@code Consumer} that performs, in sequence, this
     * operation followed by the {@code after} operation. If performing either
     * operation throws an exception, it is relayed to the caller of the
     * composed operation.  If performing this operation throws an exception,
     * the {@code after} operation will not be performed.
     *
     * @param after the operation to perform after this operation
     * @return a composed {@code Consumer} that performs in sequence this
     * operation followed by the {@code after} operation
     * @throws NullPointerException if {@code after} is null
     */
    default DbConsumer<T> andThen(DbConsumer<? super T> after) {
        Objects.requireNonNull(after);
        return (T t) -> { accept(t); after.accept(t); };
    }
}
