/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.db.layer0.variable;

import java.util.concurrent.TimeUnit;

import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.exception.InvalidVariableException;
import org.simantics.db.layer0.exception.MissingVariableValueException;
import org.simantics.db.layer0.variable.Variables.NodeStructure;
import org.simantics.simulator.variable.NodeManager;
import org.simantics.simulator.variable.exceptions.NoValueException;
import org.simantics.simulator.variable.exceptions.NodeManagerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Antti Villberg
 *
 * @param <Node>
 * @since 1.23
 */
public class NodeSupport<Node> {

	private static final Logger LOGGER = LoggerFactory.getLogger(NodeSupport.class);

	public final NodeManager<Node> manager;
	public final NodeCache<Node,Variant> valueCache;
	public final NodeCache<Node,NodeStructure> structureCache;

	public NodeSupport(NodeManager<Node> manager) {
		this(manager, 1, TimeUnit.SECONDS);
	}

	public NodeSupport(NodeManager<Node> manager, long defaultExpirationTime, TimeUnit expirationTimeUnit) {
		if (manager == null)
			throw new NullPointerException("null NodeManager");
		long ns = expirationTimeUnit.toNanos(defaultExpirationTime);
		this.manager = manager;
		this.valueCache = new NodeCache<>(ns);
		this.structureCache = new NodeCache<>(ns);
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + valueCache.hashCode();
		result = prime * result + structureCache.hashCode();
		result = prime * result + manager.hashCode();
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		NodeSupport<?> other = (NodeSupport<?>) obj;
		return valueCache.equals(other.valueCache)
				&& structureCache.equals(other.structureCache)
				&& manager.equals(other.manager);
	}

	@SuppressWarnings("rawtypes")
	private static class ValueGetter implements VariableNodeReadRunnable {

		final VariableNode n;
		final Binding binding;
		Variant result;
		Exception exception;

		public ValueGetter(VariableNode n, Binding binding) {
			this.n = n;
			this.binding = binding;
		}

		@SuppressWarnings("unchecked")
		@Override
		public void run() {
			try {
				if (binding != null) {
					Object value = n.support.manager.getValue(n.node, binding);
					if(Variables.PENDING_NODE_VALUE == value)
						result = Variables.PENDING_NODE_VALUE;
					else
						result = new Variant(binding, value);
				}
				else
					result = n.support.manager.getValue(n.node);
			} catch (NodeManagerException e) {
				exception = e;
			} catch (Exception e) {
				LOGGER.error("Error while getting node value", e);
				exception = e;
			}
		}

	}

	public Variant requestNodeValue(ReadGraph graph, VariableNode<?> node, final Binding binding) throws DatabaseException {
		Variant value = graph.syncRequest(new NodeValueRequest(node, binding));
		if(Variables.PENDING_NODE_VALUE == value && (Boolean)graph.getHintValue(ReadGraph.GRAPH_HINT_SYNCHRONOUS)) {
			// In this case a PENDING value was previously cached but now the value needs to be obtained for real.

			ValueGetter getter = new ValueGetter(node, binding);
			try {
				node.support.manager.getRealm().syncExec(getter);
			} catch (InterruptedException e) {
				LOGGER.error("Error while getting node value", e);
				throw new InvalidVariableException("External data access error " + String.valueOf(node), e);
			}

			if (getter.exception != null)
				throw new MissingVariableValueException("No value for node " + node, getter.exception);

			return getter.result;
		}
		return value; 
	}

	public void refreshCache(Node node) throws NodeManagerException {
		VariableNode<Node> variableNode = new VariableNode<>(this, node);
		NodeStructure ns = NodeStructureRequest.get(variableNode);
		structureCache.put(node, ns);
		try {
			Variant value = manager.getValue(node);
			valueCache.put(node, value);
		} catch (NoValueException e) {
			valueCache.remove(node);
		};
	}

	public void invalidateCache(Node node) throws NodeManagerException {
		structureCache.remove(node);
		valueCache.remove(node);
	}

	public void dispose() {
		valueCache.dispose();
		structureCache.dispose();
	}

}
