package org.simantics.db.layer0.migration;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.ByteBuffer;
import java.nio.LongBuffer;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.IntConsumer;

import org.eclipse.collections.api.RichIterable;
import org.eclipse.collections.api.list.ListIterable;
import org.eclipse.collections.api.multimap.list.ListMultimap;
import org.eclipse.collections.api.multimap.list.MutableListMultimap;
import org.eclipse.collections.api.multimap.set.MutableSetMultimap;
import org.eclipse.collections.impl.factory.Multimaps;
import org.eclipse.collections.impl.factory.Multimaps.MutableMultimaps.MutableListMultimapFactory;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.databoard.util.binary.BinaryFile;
import org.simantics.databoard.util.binary.DeferredBinaryFile;
import org.simantics.databoard.util.binary.DeferredBinaryFile.FileSupplier;
import org.simantics.databoard.util.binary.RandomAccessBinary;
import org.simantics.db.DirectStatements;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.uri.ResourceToPossibleURI;
import org.simantics.db.common.utils.CommonDBUtils;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.function.DbFunction;
import org.simantics.db.layer0.internal.SimanticsInternal;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.service.CollectionSupport;
import org.simantics.db.service.DirectQuerySupport;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.scl.runtime.tuple.Tuple2;
import org.simantics.scl.runtime.tuple.Tuple3;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Antti Villberg, Tuukka Lehtonen
 * @since 1.54.0
 */
public class DomainMigration {

	private static final Logger LOGGER = LoggerFactory.getLogger(DomainMigration.class);

	private static final int CHANGELOG_ENTRY_BYTE_SIZE = 8 * 5;

	/**
	 * 20000 change limit for {@link MigrationConfig#collectApplicableChanges} in-memory
	 * buffering by default.
	 */
	private static final int CHANGELOG_IN_MEMORY_THRESHOLD_SIZE = CHANGELOG_ENTRY_BYTE_SIZE * 20000;

	private static final int CHANGELOG_BUFFER_SIZE = 1 << 16;

	public static class MigrationConfig {
		/**
		 * If {@link #dryRun} is <code>true</code>, migrations will not perform any
		 * database changes during their execution but can report what changes would be
		 * made. When <code>false</code>, changes will be done immediately by the migration.
		 */
		public final boolean dryRun;

		/**
		 * Defines whether the migration should collect data for performed and failed
		 * migrations in the collections included in this class.
		 * 
		 * If the client only wants to know whether migration succeeded or failed
		 * somehow, this can be set to false by the constructor.
		 */
		public final boolean collectDetailedReportData;

		/**
		 * If {@link #dryRun} is <code>true</code> and {@link #collectApplicableChanges}
		 * is <code>true</code> the migration will collect the statement changes that
		 * would be made into the returned {@link MigrationReport}. The changes can then
		 * be applied later with
		 * {@link DomainMigration#applyChanges(WriteGraph, MigrationReport)}. If
		 * {@link #dryRun} is <code>false</code>, this option has no effect.
		 */
		public final boolean collectApplicableChanges;

		public MigrationConfig(boolean dryRun, boolean collectDetailedReportData, boolean collectApplicableChanges) {
			this.dryRun = dryRun;
			this.collectDetailedReportData = collectDetailedReportData;
			this.collectApplicableChanges = collectApplicableChanges;
		}

		@Override
		public String toString() {
			return String.format(
					"MigrationConfig [dryRun=%b, collectDetailedReportData=%b, collectApplicableChanges=%b]", // $NON-NLS-1$
					dryRun, collectDetailedReportData, collectApplicableChanges);
		}
	}

	public static interface MigrationReport {

		MigrationConfig config();

		List<Tuple2> specs();

		Resource migratedResource();

		int processedResources();

		int processedStatements();

		int migratedStatements();

		int failures();

		/**
		 * Sequential triples of migrated subject, predicate, object where both the predicate and the object were changed.
		 */
		default ListMultimap<Resource, Resource> fullyChanged() { return Multimaps.immutable.list.empty(); }
		/**
		 * Sequential pairs of migrated subject, predicate, object where only the predicate resource was changed.
		 */
		default ListMultimap<Resource, Resource> changedPredicates() { return Multimaps.immutable.list.empty(); }
		/**
		 * Sequential triples of migrated subject, predicate, object where only the object resource was changed.
		 */
		default ListMultimap<Resource, Resource> changedObjects() { return Multimaps.immutable.list.empty(); }

		default ListMultimap<Resource, Statement> failedStatements() { return Multimaps.immutable.list.empty(); }

		default ListMultimap<Resource, Statement> failedPredicates() { return Multimaps.immutable.list.empty(); }

		default ListMultimap<Resource, Statement> failedObjects() { return Multimaps.immutable.list.empty(); }
	}

	private static class MigrationReportSmall implements MigrationReport {
		private final MigrationConfig config;
		private final List<Tuple2> specs;
		private final Resource migratedResource;
		private final int processedResources;
		private final int processedStatements;
		private final int migratedStatements;
		private final int failures;
		private DeferredBinaryFile changelog;

		protected MigrationReportSmall(
				MigrationConfig config,
				List<Tuple2> specs,
				Resource migratedResource,
				int processedResources,
				int processedStatements,
				int migratedStatements,
				int failures)
		{
			this.config = config;
			this.specs = specs;
			this.migratedResource = migratedResource;
			this.processedResources = processedResources;
			this.processedStatements = processedStatements;
			this.migratedStatements = migratedStatements;
			this.failures = failures;
		}

		@Override
		public MigrationConfig config() { return config; }
		@Override
		public List<Tuple2> specs() { return specs; }
		@Override
		public Resource migratedResource() { return migratedResource; }
		@Override
		public int processedResources() { return processedResources; }
		@Override
		public int processedStatements() { return processedStatements; }
		@Override
		public int migratedStatements() { return migratedStatements; }
		@Override
		public int failures() { return failures; }
	}

	private static class MigrationReportDetailed extends MigrationReportSmall {
		private final ListMultimap<Resource, Resource> fullyChanged;
		private final ListMultimap<Resource, Resource> changedPredicates;
		private final ListMultimap<Resource, Resource> changedObjects;
		private final ListMultimap<Resource, Statement> failedStatements;
		private final ListMultimap<Resource, Statement> failedPredicates;
		private final ListMultimap<Resource, Statement> failedObjects;

		MigrationReportDetailed(
				MigrationConfig config,
				List<Tuple2> specs,
				Resource migratedResource,
				int processedResources,
				int processedStatements,
				int migratedStatements,
				int failures,
				ListMultimap<Resource, Resource> fullyChanged,
				ListMultimap<Resource, Resource> changedPredicates,
				ListMultimap<Resource, Resource> changedObjects,
				ListMultimap<Resource, Statement> failedStatements,
				ListMultimap<Resource, Statement> failedPredicates,
				ListMultimap<Resource, Statement> failedObjects)
		{
			super(config, specs, migratedResource, processedResources, processedStatements, migratedStatements, failures);
			this.fullyChanged = fullyChanged;
			this.changedPredicates = changedPredicates;
			this.changedObjects = changedObjects;
			this.failedStatements = failedStatements;
			this.failedPredicates = failedPredicates;
			this.failedObjects = failedObjects;
		}

		@Override
		public ListMultimap<Resource, Resource> fullyChanged() { return fullyChanged;}
		@Override
		public ListMultimap<Resource, Resource> changedPredicates() { return changedPredicates; }
		@Override
		public ListMultimap<Resource, Resource> changedObjects() { return changedObjects; }
		@Override
		public ListMultimap<Resource, Statement> failedStatements() { return failedStatements; }
		@Override
		public ListMultimap<Resource, Statement> failedPredicates() { return failedPredicates; }
		@Override
		public ListMultimap<Resource, Statement> failedObjects() { return failedObjects; }
	}

	private static class MigrationReportBuilder {
		final MigrationConfig config;
		final List<Tuple2> specs;
		final Resource migratedResource;

		int processedResources;
		int processedStatements;
		int migratedStatements;
		int failures;

		MutableListMultimap<Resource, Resource> fullyChanged;
		MutableListMultimap<Resource, Resource> changedPredicates;
		MutableListMultimap<Resource, Resource> changedObjects;
		MutableListMultimap<Resource, Statement> failedStatements;
		MutableListMultimap<Resource, Statement> failedPredicates;
		MutableListMultimap<Resource, Statement> failedObjects;

		DeferredBinaryFile changelog;

		MigrationReportBuilder(MigrationConfig config, Resource migratedResource, List<Tuple2> specs) {
			this.config = config;
			this.migratedResource = migratedResource;
			this.specs = specs;

			if (config.collectDetailedReportData) {
				MutableListMultimapFactory f = Multimaps.mutable.list;
				this.fullyChanged = f.empty();
				this.changedPredicates = f.empty();
				this.changedObjects = f.empty();
				this.failedStatements = f.empty();
				this.failedPredicates = f.empty();
				this.failedObjects = f.empty();
			}
			if (config.collectApplicableChanges && !config.dryRun) {
				FileSupplier fileSupplier = () -> {
					File tmp = SimanticsInternal.getTemporaryDirectory();
					return File.createTempFile("domain-migration-", ".clog", tmp);
				};
				this.changelog = new DeferredBinaryFile(fileSupplier, CHANGELOG_IN_MEMORY_THRESHOLD_SIZE, CHANGELOG_BUFFER_SIZE);
			}
		}

		MigrationReportSmall build0() {
			if (config.collectDetailedReportData) {
				return new MigrationReportDetailed(
					config,
					specs,
					migratedResource,
					processedResources,
					processedStatements,
					migratedStatements,
					failures,
					fullyChanged,
					changedPredicates,
					changedObjects,
					failedStatements,
					failedPredicates,
					failedObjects);
			}
			return new MigrationReportSmall(
					config,
					specs,
					migratedResource,
					processedResources,
					processedStatements,
					migratedStatements,
					failures
					);
		}

		MigrationReport build() {
			MigrationReportSmall r = build0();
			if (changelog != null)
				r.changelog = changelog;
			return r;
		}
}

	private static Resource browse(ReadGraph graph, Resource r, String path) throws DatabaseException {
		int beginIndex = 0;
		int endIndex = path.indexOf(URIStringUtils.NAMESPACE_PATH_SEPARATOR, beginIndex);
		while (true) {
			if (endIndex == -1) {
				return CommonDBUtils.getPossibleChild(graph, r, URIStringUtils.unescape(path.substring(beginIndex)));
			} else {
				r = CommonDBUtils.getPossibleChild(graph, r, URIStringUtils.unescape(path.substring(beginIndex, endIndex)));
				if (r == null)
					return null;
				beginIndex = endIndex+1;
				endIndex = path.indexOf(URIStringUtils.NAMESPACE_PATH_SEPARATOR, beginIndex);
			}
		}
	}

	private static Resource possibleMigratedResource(ReadGraph graph, String uri, Resource source, String sourceURI, List<Resource> targets) throws DatabaseException {
		// This is already ensured in #migrateResource
		//if (!uri.startsWith(sourceURI))
		//	return null;
		if (sourceURI.equals(uri))
			return targets.get(0);
		String path = uri.substring(sourceURI.length()+1);
		for (Resource target : targets) {
			Resource ret = browse(graph, target, path);
			if (ret != null)
				return ret;
		}
		return null;
	}

	private static final class SentinelResource implements Resource {
		@Override
		public Resource get() {
			return this;
		}
		@Override
		public int compareTo(Resource o) {
			return 0;
		}
		@Override
		public long getResourceId() {
			return 0;
		}
		@Override
		public int getThreadHash() {
			return 0;
		}
		@Override
		public boolean isPersistent() {
			return false;
		}
		@Override
		public boolean equalsResource(Resource other) {
			return false;
		}
	};

	private static final Resource NULL = new SentinelResource();
	private static final Resource FAILED = new SentinelResource();

	// Inputs 
	private final boolean debug = LOGGER.isDebugEnabled();
	private final boolean trace = LOGGER.isTraceEnabled();

	private final WriteGraph graph;
	private final MigrationConfig config;

	// Internal state

	private MigrationReportBuilder report;

	private Map<Resource, Resource> migrated;
	private Set<Resource> internalSet;
	private List<Tuple3> specsExt;
	private boolean migrationFailedForLastResource;

	private DomainMigration(WriteGraph graph, MigrationConfig config) {
		this.graph = graph;
		this.config = config;
	}

	private Resource migrateResource(Resource r) throws DatabaseException {
		migrationFailedForLastResource = false;

		Resource alreadyMigrated = migrated.get(r);
		if (alreadyMigrated != null) {
			if (NULL == alreadyMigrated)
				return null;
			if (FAILED == alreadyMigrated) {
				migrationFailedForLastResource = true;
				return null;
			}
			return alreadyMigrated;
		}

		if (internalSet.contains(r)) {
			migrated.put(r, NULL);
			return null;
		}

		Resource result = null;

		String uri = graph.syncRequest(new ResourceToPossibleURI(r), TransientCacheAsyncListener.instance());
		if (uri != null) {
			boolean resourceInSomeMigratedSource = false;
			for (Tuple3 spec : specsExt) {
				String sourceContainerURI = (String) spec.c0;
				if (!uri.startsWith(sourceContainerURI))
					continue;
				resourceInSomeMigratedSource = true;

				Resource sourceContainer = (Resource) spec.c1;
				@SuppressWarnings("unchecked")
				List<Resource> targetContainers = (List<Resource>) spec.c2;

				result = possibleMigratedResource(graph, uri, sourceContainer, sourceContainerURI, targetContainers);
				if (result != null)
					break;
			}
			if (result == null && resourceInSomeMigratedSource) {
				migrationFailedForLastResource = true;
				migrated.put(r, FAILED);
				if (debug) {
					LOGGER.debug("Failed to migrate external resource {}, no correspondence found in any target container",
							NameUtils.getURIOrSafeNameInternal(graph, r));
				}
				return null;
			}
		}

		if (result == null) {
			migrated.put(r, NULL);
			return null;
		} else {
			migrated.put(r, result);
			return result;
		}
	}

	private MigrationReport migrateDomainSpec(Resource r, List<Tuple2> specs) throws DatabaseException {
		report = new MigrationReportBuilder(config, r, specs);

		try {
			final boolean dryRun = config.dryRun;
			final CollectionSupport cs = graph.getService(CollectionSupport.class);
			final DirectQuerySupport dqs = graph.getService(DirectQuerySupport.class);
			final Collection<Resource> internalList = Layer0Utils.domainResources(graph, r);
	
			this.internalSet = cs.getResourceSet(graph, internalList);
			this.migrated = cs.createMap(Resource.class);
	
			final int count = internalList.size();
			int done = 0;
	
			List<Resource> temp = null;
			if (config.collectDetailedReportData) {
				temp = new ArrayList<>();
				temp.add(null);
				temp.add(null);
			}
	
			byte[] changebuf = null;
			LongBuffer changebufl = null;
			if (report.changelog != null) {
				changebuf = new byte[CHANGELOG_ENTRY_BYTE_SIZE];
				changebufl = ByteBuffer.wrap(changebuf).asLongBuffer();
				changebufl.limit(CHANGELOG_ENTRY_BYTE_SIZE / 8);
			}
	
			report.processedResources = count;
	
			this.specsExt = new ArrayList<>();
			for (Tuple2 spec : specs) {
				specsExt.add(new Tuple3(graph.getURI((Resource)spec.c0), spec.c0, spec.c1));
			}
	
			for (Resource internal : internalList) {
				DirectStatements ds = dqs.getDirectStatements(graph, internal);
				for (Statement stm : ds) {
					Resource predicate = stm.getPredicate();
					Resource object = stm.getObject();
	
					Resource migratedPredicate = migrateResource(predicate);
					boolean predicateFailed = migrationFailedForLastResource;
					Resource migratedObject = migrateResource(object);
					boolean objectFailed = migrationFailedForLastResource;
	
					boolean predicateMigrated = migratedPredicate != null;
					boolean objectMigrated = migratedObject != null;
					if (predicateMigrated || objectMigrated) {
						if (migratedPredicate == null)
							migratedPredicate = predicate;
						if (migratedObject == null)
							migratedObject = object;
						if (config.collectDetailedReportData) {
							temp.set(0, migratedPredicate);
							temp.set(1, migratedObject);
							if (predicateMigrated && objectMigrated) {
								report.fullyChanged.putAll(internal, temp);
							} else if (predicateMigrated) {
								report.changedPredicates.putAll(internal, temp);
							} else /*(objectMigrated)*/ {
								report.changedObjects.putAll(internal, temp);
							}
						}
						if (report.changelog != null) {
							changebufl.rewind();
							changebufl.put(internal.getResourceId());
							changebufl.put(predicate.getResourceId());
							changebufl.put(object.getResourceId());
							changebufl.put(migratedPredicate.getResourceId());
							changebufl.put(migratedObject.getResourceId());
							try {
								report.changelog.write(changebuf);
							} catch (IOException e) {
								throw new DatabaseException("Failed to write migration change log contents to disk", e);
							}
						}
						if (!dryRun) {
							graph.deny(internal, predicate, object);
							graph.claim(internal, migratedPredicate, migratedObject);
							if (trace) {
								LOGGER.trace("migrate: {} {} {}",
										NameUtils.getURIOrSafeNameInternal(graph, internal),
										NameUtils.getURIOrSafeNameInternal(graph, migratedPredicate),
										NameUtils.getURIOrSafeNameInternal(graph, migratedObject));
							}
						}
						report.migratedStatements++;
					} else {
						if (predicateFailed || objectFailed) {
							report.failures++;
							if (config.collectDetailedReportData) {
								if (predicateFailed && objectFailed) {
									report.failedStatements.put(internal, stm);
								} else if (predicateFailed) {
									report.failedPredicates.put(internal, stm);
								} else {
									report.failedObjects.put(internal, stm);
								}
							}
						}
					}
	
					report.processedStatements++;
				}
				done++;
				if (trace && done % 1000L == 0L)
					LOGGER.trace("processed {}/{}", done, count);
			}
	
			return report.build();
		} finally {
			try {
				// Ensure no files are left open.
				if(report.changelog != null)
					report.changelog.close();
			} catch (IOException e) {
				LOGGER.error("Failed to close collected changelog file", e);
			}
		}
	}

	public static MigrationReport migrateDomainWithSpecs(WriteGraph graph, MigrationConfig config, Resource r, List<Tuple2> specs) throws DatabaseException {
		return new DomainMigration(graph, config).migrateDomainSpec(r, specs);
	}

	public static MigrationReport migrateDomain(WriteGraph graph, MigrationConfig config, Resource r, Resource sourceContainer, Resource targetContainer) throws DatabaseException {
		return migrateDomainWithSpecs(graph, config, r, Collections.singletonList(
				new Tuple2(sourceContainer, Collections.singletonList(targetContainer))));
	}

	private static void dumpReportSmall(ReadGraph graph, MigrationReport r, PrintWriter out) throws DatabaseException {
		out.append("# Migration report for ").println(NameUtils.getURIOrSafeNameInternal(graph, r.migratedResource()));
		if (r.config().dryRun) {
			out.println("# Dry-run mode - nothing changed");
		}
		out.println("Specifications:");
		for (Tuple2 t : r.specs()) {
			Resource src = (Resource) t.c0;
			@SuppressWarnings("unchecked")
			List<Resource> tgts = (List<Resource>) t.c1;
			out.append("\t").append(graph.getPossibleURI(src)).append(" -> [");
			boolean first = true;
			for (Resource tgt : tgts) {
				if (!first)
					out.append(", ");
				first = false;
				out.append(graph.getPossibleURI(tgt));
			}
			out.println("]");
		}
		out.append("Processed resources:  ").println(r.processedResources());
		out.append("Processed statements: ").println(r.processedStatements());
		out.append("Migrated statements:  ").println(r.migratedStatements());
		out.append("Failed migrations:    ").println(r.failures());
	}

	private static void reportChangedStatements(ReadGraph graph, ListMultimap<Resource, Resource> rs, PrintWriter out, IntConsumer header) throws DatabaseException {
		int rss = rs.size();
		if (rss > 0) {
			header.accept(rss);
			int sc = 0;
			for (Resource s : rs.keysView()) {
				ListIterable<Resource> vs = rs.get(s);
				int sz = vs.size();
				out.append("\t[" + sc + "]\t").println(NameUtils.getURIOrSafeNameInternal(graph, s));
				for (int i = 0; i < sz; i += 2) {
					Resource mp = vs.get(i);
					Resource mo = vs.get(i+1);
					out.append("\t\t[" + (i/2) + "]\t").println(NameUtils.getURIOrSafeNameInternal(graph, mp));
					out.append("\t\t\t\t").println(NameUtils.getURIOrSafeNameInternal(graph, mo));
				}
				++sc;
			}
		}
	}

	private static void reportFailedStatements(ReadGraph graph, ListMultimap<Resource, Statement> rs, PrintWriter out, IntConsumer header) throws DatabaseException {
		int rss = rs.size();
		if (rss > 0) {
			header.accept(rss);
			int sc = 0;
			for (Resource s : rs.keysView()) {
				out.append("\t[" + sc + "]\t").println(NameUtils.getURIOrSafeNameInternal(graph, s));
				int pc = 0;
				for (Statement stm : rs.get(s)) {
					out.append("\t\t[" + pc + "]\t").println(NameUtils.getURIOrSafeNameInternal(graph, stm.getPredicate()));
					out.append("\t\t\t\t").println(NameUtils.getURIOrSafeNameInternal(graph, stm.getObject()));
					++pc;
				}
				++sc;
			}
		}
	}

	public static void dumpReport(ReadGraph graph, MigrationReport r, PrintWriter out) throws DatabaseException { 
		dumpReportSmall(graph, r, out);
		reportChangedStatements(graph, r.fullyChanged(), out, size -> {
			out.println("Subjects with changed statements [" + size/2 + "]:");
		});
		reportChangedStatements(graph, r.changedPredicates(), out, size -> {
			out.println("Subjects with changed predicates [" + size/2 + "]:");
		});
		reportChangedStatements(graph, r.changedObjects(), out, size -> {
			out.println("Subjects with changed objects [" + size/2 + "]:");
		});
		reportFailedStatements(graph, r.failedStatements(), out, size -> {
			out.println("Subjects with failed statement migrations [" + size + "]:");
		});
		reportFailedStatements(graph, r.failedPredicates(), out, size -> {
			out.println("Subjects with failed predicate migrations [" + size + "]:");
		});
		reportFailedStatements(graph, r.failedObjects(), out, size -> {
			out.println("Subjects with failed object migrations [" + size + "]:");
		});
	}

	private static <T> void forPairs(RichIterable<T> ri, BiConsumer<T, T> c) {
		T r1 = null;
		for (T r : ri) {
			if (r1 == null) {
				r1 = r;
			} else {
				c.accept(r1, r);
				r1 = null;
			}
		}
	}

	private static void forStatements(RichIterable<Statement> ri, BiConsumer<Resource, Resource> c) {
		ri.forEach(s -> c.accept(s.getPredicate(), s.getObject()));
	}

	private static List<Resource> sortedList(CollectionSupport cs, Collection<Resource> c) {
		return c.isEmpty() ? Collections.emptyList() : cs.asSortedList(c);
	}

	public static void dumpReport2(ReadGraph graph, MigrationReport r, PrintWriter out) throws DatabaseException {
		dumpReportSmall(graph, r, out);
		if (!r.config().collectDetailedReportData)
			return;

		CollectionSupport cs = graph.getService(CollectionSupport.class);
		Set<Resource> subjects = cs.createSet(
				r.fullyChanged().size()
				+ r.changedPredicates().size()
				+ r.changedObjects().size()
				);
		r.fullyChanged().forEachKey(subjects::add);
		r.changedPredicates().forEachKey(subjects::add);
		r.changedObjects().forEachKey(subjects::add);

		Set<Resource> prs = cs.createSet();
		Set<Resource> changedPredicate = cs.createSet();
		Set<Resource> changedObject = cs.createSet();
		MutableSetMultimap<Resource, Resource> po = Multimaps.mutable.set.empty();

		int ssz = subjects.size();
		if (ssz > 0) {
			out.println("Subjects with changed statements [" + subjects.size() + "]:");
			int sc = 0;
			for (Resource s : sortedList(cs, subjects)) {
				out.append("S[" + sc + "]\t").println(NameUtils.getURIOrSafeNameInternal(graph, s));

				po.clear();
				prs.clear();
				changedPredicate.clear();
				changedObject.clear();

				forPairs(r.fullyChanged().get(s), (p,o) -> {
					po.put(p, o);
					prs.add(p);
					changedPredicate.add(p);
					changedObject.add(o);
				});
				forPairs(r.changedPredicates().get(s), (p,o) -> {
					po.put(p, o);
					prs.add(p);
					changedPredicate.add(p);
				});
				forPairs(r.changedObjects().get(s), (p,o) -> {
					po.put(p, o);
					prs.add(p);
					changedObject.add(o);
				});

				int pc = 0;
				for (Resource p : sortedList(cs, prs)) {
					out.append('\t').append(changedPredicate.contains(p) ? "* " : "  ").append("P[" + pc + "]\t").println(NameUtils.getURIOrSafeNameInternal(graph, p));
					int oc = 0;
					for (Resource o : sortedList(cs, po.get(p))) {
						out.append("\t\t").append(changedObject.contains(o) ? "* " : "  ").append("O[" + oc + "]\t").println(NameUtils.getURIOrSafeNameInternal(graph, o));
						++oc;
					}
					++pc;
				}

				++sc;
			}
		}

		subjects.clear();
		r.failedStatements().forEachKey(subjects::add);
		r.failedPredicates().forEachKey(subjects::add);
		r.failedObjects().forEachKey(subjects::add);

		ssz = subjects.size();
		if (ssz > 0) {
			out.println("Subjects with statements that failed migration [" + subjects.size() + "]:");
			int sc = 0;
			for (Resource s : sortedList(cs, subjects)) {
				out.append("F S[" + sc + "]\t").println(NameUtils.getURIOrSafeNameInternal(graph, s));

				po.clear();
				prs.clear();
				changedPredicate.clear();
				changedObject.clear();

				forStatements(r.failedStatements().get(s), (p,o) -> {
					po.put(p, o);
					prs.add(p);
					changedPredicate.add(p);
					changedObject.add(o);
				});
				forStatements(r.failedPredicates().get(s), (p,o) -> {
					po.put(p, o);
					prs.add(p);
					changedPredicate.add(p);
				});
				forStatements(r.failedObjects().get(s), (p,o) -> {
					po.put(p, o);
					prs.add(p);
					changedObject.add(o);
				});

				int pc = 0;
				for (Resource p : sortedList(cs, prs)) {
					out.append('\t').append(changedPredicate.contains(p) ? "F " : "  ").append("P[" + pc + "]\t").println(NameUtils.getURIOrSafeNameInternal(graph, p));
					int oc = 0;
					for (Resource o : sortedList(cs, po.get(p))) {
						out.append("\t\t").append(changedObject.contains(o) ? "F " : "  ").append("O[" + oc + "]\t").println(NameUtils.getURIOrSafeNameInternal(graph, o));
						++oc;
					}
					++pc;
				}

				++sc;
			}
		}
	}

	public static String dumpReport(ReadGraph graph, MigrationReport r) throws DatabaseException {
		StringWriter sw = new StringWriter(64*1024);
		try (PrintWriter out = new PrintWriter(sw)) {
			dumpReport(graph, r, out);
			return sw.toString();
		}
	}

	public static String dumpReport2(ReadGraph graph, MigrationReport r) throws DatabaseException {
		StringWriter sw = new StringWriter(64*1024);
		try (PrintWriter out = new PrintWriter(sw)) {
			dumpReport2(graph, r, out);
			return sw.toString();
		}
	}

	public static void applyChanges(WriteGraph graph, MigrationReport report) throws DatabaseException {
		MigrationReportSmall r = (MigrationReportSmall) report;
		if (r.changelog == null) {
			throw new IllegalArgumentException("Migration report was not created with MigrationConfig.collectApplicableChanges set to true or the migrations have already been applied.");
		}

		Map<Resource, Resource> inverseMap = new HashMap<>();
		DbFunction<Resource, Resource> inverseFunction = p -> {
			Resource inv = inverseMap.get(p);
			if (inv != null) {
				return inv != NULL ? inv : null;
			}
			inv = graph.getPossibleInverse(p);
			inverseMap.put(p, inv != null ? inv : NULL);
			return inv;
		};

		SerialisationSupport ss = graph.getService(SerialisationSupport.class);

		File clogFile = null;
		RandomAccessBinary rab = null;
		long dataLength = 0L;
		try {
			if (r.changelog.isInMemory()) {
				rab = r.changelog.getMemory();
				dataLength = rab.position();
				rab.position(0);
			} else {
				clogFile = r.changelog.getFile();
				rab = new BinaryFile(clogFile, CHANGELOG_BUFFER_SIZE);
				dataLength = rab.length();
			}

			long changes = dataLength / CHANGELOG_ENTRY_BYTE_SIZE;
			try (RandomAccessBinary _rab = rab) {
				for (int i = 0; i < changes; ++i) {
					long s = rab.readLong();
					long p = rab.readLong();
					long o = rab.readLong();
					long mp = rab.readLong();
					long mo = rab.readLong();

					Resource sr = ss.getResource(s);
					Resource pr = ss.getResource(p);
					Resource or = ss.getResource(o);
					Resource mpr = mp == p ? pr : ss.getResource(mp);
					Resource mor = mo == o ? or : ss.getResource(mo);
					Resource prinv = inverseFunction.apply(pr);
					Resource mprinv = mp == p ? prinv : inverseFunction.apply(mpr);

					graph.deny(sr, pr, prinv, or);
					graph.claim(sr, mpr, mprinv, mor);
				}
			}

			// Mark changelog applied
			r.changelog = null;

		} catch (IOException e) {
			throw new DatabaseException("Failed to read migration change log file during its application", e);
		} finally {
			try {
				if (r.changelog != null && r.changelog.isInMemory()) {
					r.changelog.position(dataLength);
				}
			} catch (IOException e) {
				// Log and ignore on purpose
				LOGGER.error("Failed to reset original memory buffer position", dataLength, e);
			}

			if (clogFile != null) {
				try {
					Files.delete(clogFile.toPath());
				} catch (IOException e) {
					// Log and ignore on purpose.
					LOGGER.error("Failed to delete temporary file {}", clogFile, e);
				}
			}
		}
	}

}
