/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.adapter.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.util.URIStringUtils;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.service.XSupport;
import org.simantics.graph.db.AbstractImportAdvisor2;
import org.simantics.graph.db.ImportAdvisors;
import org.simantics.graph.db.MissingDependencyException;
import org.simantics.graph.db.TransferableGraphImporter;
import org.simantics.graph.representation.Root;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.datastructures.Triple;

public class SharedOntologyImportAdvisor extends AbstractImportAdvisor2 {

    private Layer0 L0;

    private Map<Root, Triple<Resource, Resource, List<String>>> toCreate = new HashMap<Root, Triple<Resource, Resource, List<String>>>();
    private Map<Root, Resource> created = new HashMap<Root, Resource>();
    private Pair<Boolean,Boolean> mode;

    final private boolean published;

    final private Map<String,Object> context;

    public SharedOntologyImportAdvisor(boolean published) {
        this(Collections.<String,Object>emptyMap(), published);
    }

    public SharedOntologyImportAdvisor(Map<String,Object> context, boolean published) {
        this.context = context;
        this.published = published;
    }

    private Resource getNames(ReadGraph graph, String uri, List<String> names) throws DatabaseException {

        if("http://".equals(uri) || "http:/".equals(uri)) {
            return graph.getRootLibrary();
        }

        final String[] parts = URIStringUtils.splitURI(uri);
        if(parts != null && parts.length == 2) {
            Resource base = graph.getPossibleResource(parts[0]);
            if(base == null) base = getNames(graph, parts[0], names);
            names.add(parts[1]);
            return base;
        }

        return null;

    }


    @Override
    public Resource analyzeRoot(ReadGraph graph, Root r) throws DatabaseException {

        L0 = Layer0.getInstance(graph);

        String uri = r.name;
        Map<String,String> renameMap = (Map<String,String>)context.get(ImportAdvisors.RENAME_MAP);
        if(renameMap != null) {
            String renamed = renameMap.get(uri);
            if(renamed != null) uri = renamed;
        }

        Resource existing = graph.getPossibleResource(uri);
        if(existing != null) {
            if(graph.isInstanceOf(existing, L0.ExternalEntity)) {
                created.put(r, existing);
                addRootInfo(r, r.name, existing);
                return null;
            } else {
                throw new DatabaseException("Shared library " + uri + " exists already.");
            }
        }

        Resource type = graph.getPossibleResource(r.type);
        if(type == null) { 
            Set<String> types = new HashSet<String>();
            types.add(r.type);
            throw new MissingDependencyException(types);
        }
        //if(type == null) throw new DatabaseException("Shared library type " + r.type + " was not found.");

        List<String> namesToCreate = new ArrayList<String>();
        Resource base = getNames(graph, uri, namesToCreate);
        toCreate.put(r, Triple.make(base, type, namesToCreate));

        return null;

    }

    @Override
    public Resource getTarget() {
        return null;
    }

    @Override
    public Resource createRoot(WriteOnlyGraph graph, Root r) throws DatabaseException {
        return createRoot(graph, r, null);
    }

    @Override
    public Resource createRoot(WriteOnlyGraph graph, Root root, Resource resource) throws DatabaseException {
        return created.get(root);
    }

    @Override
    public void beforeWrite(WriteOnlyGraph graph, TransferableGraphImporter process)
            throws DatabaseException {

        XSupport xs = graph.getService(XSupport.class);

        mode = xs.getServiceMode();

        try {

            graph.markUndoPoint();

            if(published) {
                xs.setServiceMode(true, true);
            } else {
                xs.setServiceMode(true, mode.second);
            }

            graph.setClusterSet4NewResource(graph.getRootLibrary());
            graph.flushCluster();

            for(RootInfo info : getRootInfo()) {
                // At this stage these are existing L0.ExternalEntity instances that are now being imported.
                graph.deny(info.resource, L0.InstanceOf, null, L0.ExternalEntity, null);
            }

            for(Map.Entry<Root, Triple<Resource, Resource, List<String>>> entry : toCreate.entrySet()) {

                Triple<Resource, Resource, List<String>> recipe = entry.getValue();

                Resource base = recipe.first;
                for(int i=0;i<recipe.third.size()-1;i++) {
                    Resource lib = graph.newResource();
                    graph.claim(lib, L0.InstanceOf, null, L0.Library);
                    graph.addLiteral(lib, L0.HasName, L0.NameOf, URIStringUtils.unescape( recipe.third.get(i) ), Bindings.STRING);
                    graph.claim(base, L0.ConsistsOf, L0.PartOf, lib);
                    base = lib;
                }

                Resource lib = graph.newResource();
                graph.newClusterSet(lib);
                graph.setClusterSet4NewResource(lib);
                graph.claim(lib, L0.InstanceOf, null, recipe.second);
                String name = URIStringUtils.unescape( recipe.third.get(recipe.third.size()-1) );
                graph.addLiteral(lib, L0.HasName, L0.NameOf, name, Bindings.STRING);
                graph.claim(base, L0.ConsistsOf, L0.PartOf, lib);

                addRootInfo(entry.getKey(), name, lib);

                created.put(entry.getKey(), lib);

            }

            graph.flushCluster();

        } catch (Exception e) {

            xs.setServiceMode(mode.first, mode.second);

            throw new DatabaseException(e);

        }

    }

    @Override
    public void afterWrite(WriteOnlyGraph graph, TransferableGraphImporter process)
            throws DatabaseException {

        String name = getRootInfo().iterator().next().name;
        Resource resource = getRootInfo().iterator().next().resource;
        Layer0Utils.addCommentMetadata(graph, "Imported Shared Ontology " + name + " " + resource);

        XSupport xs = graph.getService(XSupport.class);
        xs.setServiceMode(mode.first, mode.second);

        graph.setClusterSet4NewResource(graph.getRootLibrary());
        graph.flushCluster();

    }

    @Override
    public boolean allowImmutableModifications() {
        return false;
    }

    @Override
    public Resource createChild(WriteOnlyGraph graph, TransferableGraphImporter process, Resource parent,
            String name) throws DatabaseException {
        return process.createChild(graph, parent, null, name);
    }

    @Override
    public Resource createChild(WriteOnlyGraph graph, TransferableGraphImporter process, Resource parent, Resource child,
            String name) throws DatabaseException {
        return process.createChild(graph, parent, child, name);
    }

}
