package org.simantics.db.impl.query;

import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.Session;
import org.simantics.db.common.SessionThread;
import org.simantics.db.impl.query.QueryProcessor.SessionTask;
import org.simantics.db.impl.query.QueryProcessor.ThreadState;
import org.slf4j.LoggerFactory;

public class QueryThread extends Thread implements SessionThread {

	private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(QueryThread.class);

	boolean disposed = false;
	private Semaphore exited = new Semaphore(0);

	final int index;

	private Session session;
	private QuerySupport querySupport;
	private final QueryProcessor processor;
	private final Semaphore requests;

	final private Object querySupportLock;
	final private int THREADS;
	final private AtomicInteger sleepers;
	final private ThreadState[] threadStates;

	public final static ThreadLocal<QueryThread> threadLocal = new ThreadLocal<>(); 
	
	public QueryThread(Session session, QueryProcessor processor, int index, String name) {
		super(QueryProcessor.QueryThreadGroup, null, name);
		this.session = session;
		this.processor = processor;
		this.index = index;
		querySupportLock = processor.querySupportLock;
		THREADS = processor.THREADS;
		sleepers = processor.sleepers;
		querySupport = processor.querySupport;
		threadStates = processor.threadStates;
		requests = processor.requests;
	}

	synchronized void dispose() {

		disposed = true;
		
		try {
			exited.acquire();
		} catch (InterruptedException e) {
			LOGGER.error("dispose was interrupted", e);
		}

		session = null;
		querySupport = null;

	}

	boolean isDisposed() { 
		return disposed;
	}

	public Session getSession() {
		return session;
	}
	
	public int getIndex() {
		return index;
	}

	SessionTask newTask() {

		try {

			while(true) {

				synchronized (querySupportLock) {

					SessionTask task = processor.scheduling.pumpTask();
					if(task != null)
						return task;

					// We are the last one awake
					if(sleepers.incrementAndGet() == THREADS) {

						// Do not indicate sleeping yet
						sleepers.decrementAndGet();
						// Ceased can create new own tasks
						if(querySupport == null) System.err.println("null qs");
						querySupport.ceased(index);

						task = processor.scheduling.pumpTask();
						if(task != null)
							return task;

						// OK, now we are going to sleep
						sleepers.incrementAndGet();

					}

				}

				// Nope nothing. Sleep & wait
				// Whoever releases this calls sleepers.decrementAndGet()

				// We are done
				if(isDisposed()) {
					threadStates[index] = ThreadState.DISPOSED;
					return null;
				}

				
				threadStates[index] = ThreadState.SLEEP;
				
				requests.acquire();

				sleepers.decrementAndGet();

				// We are done
				if(isDisposed()) {
					threadStates[index] = ThreadState.DISPOSED;
					return null;
				}

				threadStates[index] = ThreadState.RUN;

			}

		} catch (InterruptedException e) {

			LOGGER.error("Query handling (newTasks) was interrupted", e);
			throw new RuntimeException("Querying was interrupted.", e);

		}

	}

	@Override
	public void run() {

		threadLocal.set(this);
		
		QuerySupport support = this.querySupport;

		try {

			while (true) {

				SessionTask task = newTask();
				if(task == null)
					return;
				
				task.run(index);
				
			}

		} catch (Throwable t) {

			LOGGER.error("FATAL BUG: QueryThread task processing caused unexpected exception.", t);
			support.exit(t);

		} finally {
			
			exited.release();

		}

	}

}