package org.simantics.db.impl.query;

import org.simantics.db.impl.query.QueryProcessor.QueryCollectorSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class QueryCollectorImpl implements QueryProcessor.QueryCollector {
	
	private static final Logger LOGGER = LoggerFactory.getLogger(QueryCollectorImpl.class);

	private static final boolean DEBUG = false;

	private final QueryProcessor queryProcessor;

	final private QueryCollectorSupport support;

	/*
	 * Set to true: at end of iteration if moreAll is false
	 * Set to false: upon gc start and whenever a prospect is added.
	 * 
	 */
	boolean doneAll = false;
	/*
	 * Set to true: upon gc start and whenever a prospect is added.
	 * Set to false: at end of entry iteration
	 * => if this is already false at end of entry iteration => no more work to be done
	 */
	int moreAll = 0;
	
	/*
	 * Change propagation is in progress
	 */
	boolean propagate = true;
	
	
	private static final int COLLECT_N = 1000;
	
	private long spent = 0;
	
	QueryCollectorImpl(QueryProcessor queryProcessor, QueryCollectorSupport support) {
		this.queryProcessor = queryProcessor;
		this.support = support;
	}
	
	@Override
	public void collect(int youngTarget, int allowedTimeInMs) {

		// Flush listener registrations to prevent the collector from trashing
		// listeners that are still queued up waiting to be registered.
		queryProcessor.listening.stopThreading();
		collect_(youngTarget, allowedTimeInMs);
		queryProcessor.listening.startThreading();
		
	}

	public void collect_(int youngTarget, int allowedTimeInMs) {

		long start = System.nanoTime();

		// Initialize support for new run
		// If support returns 0 we are starting from 0
		boolean requestFlush = youngTarget == 0;
		boolean newData = support.start(requestFlush); 
		if(newData) {
			
			moreAll = 0;

			// We monitor all prospects here
			CacheEntryBase prospect = support.iterate(0);
			while(prospect != null) {
				if(prospect.isDiscarded()) {
					support.remove();
					propagate = true;
				} else {
					CacheEntry parent = queryProcessor.listening.firstParentNotDiscarded(prospect);
					if(parent == null) {
						tryCollect(prospect);
					} else {
						short nextLevel = parent.getLevel();
						support.setLevel(prospect, ++nextLevel);
					}
				}
				prospect = support.iterate(0);
			}

			// Refresh current size
			int size = support.calculateCurrentSize();
			long act = support.getActivity();
			long actPct = size > 0 ? 100*act / size : 0;

			// If no prospects were collected and there is no query activity we can stop here
			if(!propagate && actPct < youngTarget) {
				if(DEBUG) {
					long elapsed = System.nanoTime() - start;
					LOGGER.info("early exit after scanning prospects size=" + size + " activity=" + act + " (" + actPct + "%) in " + (1e-6*elapsed) + "ms.");
				}
				return; 
			}
			
		}
		
		long test = (long)allowedTimeInMs*1000000;

		start();
		
		while(true) {
			
			int size = support.getCurrentSize();
			
			long elapsed = System.nanoTime()-start;
			boolean timeCondition = elapsed > test; 

			if(doneAll || timeCondition) {
				
				spent += elapsed;
				
				if(DEBUG) {
					long activity = support.getActivity();
					LOGGER.info("Query collector used " + 1e-9*elapsed + "s total queries: " + size + " query updates: " + activity + " spent: " + (double)spent*1e-9);
				}
				
				return;
				
			}

			// Iterate all entries and update sets
			if(!doneAll) {
				for(int i=0;i<COLLECT_N;i++) {
					CacheEntryBase entry = support.iterate(Integer.MAX_VALUE);
					if(entry == null) {
						if(DEBUG) {
							LOGGER.info("finished iteration, moreAll=" + moreAll);
						}
						if(moreAll < 1000) {
							doneAll = true;
							propagate = false;
						}
						moreAll = 0;
						break;
					}
					

					CacheEntry<?> parent = queryProcessor.listening.firstParentNotDiscarded(entry);
					if(parent == null) {
					
						boolean collected = tryCollect(entry);
						if(!collected) {
							entry.setLevel((short)0);
						}
						
					} else {
						queryProcessor.listening.pruneParentSet(entry);
						short nextLevel = parent.getLevel();
						support.setLevel(entry, ++nextLevel);
						
					}
					
				}
			}

		}

	}

	private boolean tryCollect(CacheEntry entry) {
		if (!queryProcessor.listening.hasListenerAfterDisposing(entry))
			if(entry.shouldBeCollected()) {
				queryProcessor.removeQuery(entry);
				support.remove();
				propagate = true;
				moreAll++;
				doneAll = false;
				return true;
			}
		return false;
	}

	private void start() {
		moreAll = 1;
		doneAll = false;
	}

}