/*******************************************************************************
 * Copyright (c) 2018, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;

import org.simantics.utils.FileUtils;

public class CodeGen {

    int indent = 4;

    class GenerationInfo {
        String p1;
        String keyName;
        String keyToId;
        String procedure;
        boolean useNeedsToBlock;
        String entryCreationArguments;
        GenerationInfo(String p1, String keyName, String keyToId, String p4, String procedure, String entryCreationArguments, boolean useNeedsToBlock) {
            this.p1 = p1;
            this.keyName = keyName;
            this.keyToId = keyToId;
            this.procedure = procedure;
            this.useNeedsToBlock = useNeedsToBlock;
            this.entryCreationArguments = entryCreationArguments;
        }
    }

    GenerationInfo signatureR1RelationInfo = new GenerationInfo  ( "int r", "r", "keyR", "long", "InternalProcedure<RelationInfo>", "", false );
    GenerationInfo signatureR1Bytes = new GenerationInfo  ( "int r", "r", "keyR", "long", "InternalProcedure<byte[]>", "", false );
    GenerationInfo signatureR1IntSet = new GenerationInfo  ( "int r", "r", "keyR", "long", "InternalProcedure<IntSet>", "", false );
    GenerationInfo signatureR1IP = new GenerationInfo  ( "int r", "r", "keyR", "long", "IntProcedure", "", false );
    GenerationInfo signatureR2IP = new GenerationInfo  ( "int r1, int r2", "r1,r2", "keyR2", "long", "IntProcedure", "", false );
    GenerationInfo signatureR2TIP = new GenerationInfo  ( "int r1, int r2", "r1,r2", "keyR2", "long", "TripleIntProcedure", "", false );
    GenerationInfo signatureID1 = new GenerationInfo  ( "String id", "id", "keyID", "String", "InternalProcedure<Integer>", "", false );
    GenerationInfo signatureID2 = new GenerationInfo  ( "String id", "id", "keyID", "String", "InternalProcedure<TObjectIntHashMap<String>>", "", false );
    GenerationInfo signatureChildMap = new GenerationInfo  ( "int r", "r", "keyR", "long", "InternalProcedure<ObjectResourceIdMap<String>>", "", false );
    GenerationInfo signatureRead = new GenerationInfo  ( "Read<?> r", "r", "id", "long", "AsyncProcedure", "", true );
    GenerationInfo signatureAsyncRead = new GenerationInfo  ( "AsyncRead<?> r", "r", "id", "long", "AsyncProcedure", "", true );
    GenerationInfo signatureMultiRead = new GenerationInfo  ( "MultiRead<?> r", "r", "id", "long", "SyncMultiProcedure", "", false );
    GenerationInfo signatureAsyncMultiRead = new GenerationInfo  ( "AsyncMultiRead<?> r", "r", "id", "long", "AsyncMultiProcedure", "", false );
    GenerationInfo signatureExternalRead = new GenerationInfo  ( "ExternalRead<?> r", "r", "id", "long", "AsyncProcedure", ", graph", false );

    private void line(StringBuilder content, String line) {
        for(int i=0;i<indent;i++)
            content.append(" ");
        content.append(line);
        content.append("\n");
    }

    public void generateQuery(StringBuilder content, String clazz, GenerationInfo signature, boolean runnerShortcut, boolean genReturn) {
        generateGetOrCreate(content, clazz, signature);
        generateRemove(content, clazz, signature);
        generateRunner(content, clazz, signature, runnerShortcut, genReturn);
    }

    public void generateRunner(StringBuilder content, String clazz, GenerationInfo signature, boolean shortcut, boolean genReturn) {

        line(content, "public static " + (genReturn ? "Object" : "void") + " runner" + clazz + "(ReadGraphImpl graph, " + signature.p1 + ", CacheEntry<?> parent, ListenerBase listener, final " + signature.procedure + " procedure" + (signature.useNeedsToBlock ? ", boolean needsToBlock" : "") + ") throws DatabaseException {");
        line(content, "    QueryCache cache  = graph.processor.cache;");
        if(shortcut) {
            line(content, "    if(parent == null && listener == null && !shouldCache(graph, " + signature.keyName + ")) {");
            line(content, "        if (SINGLE) {");
            line(content, "            " + clazz + " e = cache.peek" + clazz + "(" + signature.keyName + ");");
            line(content, "            if (e != null && e.isReady()) {");
            line(content, "                " + (genReturn ? "return " : "") + "e.performFromCache(graph, procedure);");
            if(!genReturn) line(content, "                return;");
            line(content, "            }");
            line(content, "        }");
            line(content, "        " + (genReturn ? "return " : "") + clazz + ".computeForEach(graph, " + signature.keyName + ", null, procedure" + (signature.useNeedsToBlock ? ", needsToBlock" : "") + ");");
            if(!genReturn) line(content, "        return;");
            line(content, "    }");
        }
        if(signature.useNeedsToBlock) {
            line(content, "    " + clazz + " entry = (" + clazz + ")cache.getOrCreate" + clazz + "(graph, " + signature.keyName + ", needsToBlock);");
            line(content, "    if(entry == null) {");
            line(content, "      graph.asyncBarrier.inc();");
            line(content, "      graph.processor.scheduleNow(new SessionTask() {");
            line(content, "        @Override");
            line(content, "        public void run0(int thread) {");
            line(content, "          try {");
            line(content, "            runner" + clazz + "(graph, r, parent, listener, procedure, needsToBlock);");
            line(content, "            graph.asyncBarrier.dec();");
            line(content, "          } catch (DatabaseException e) {");
            line(content, "            LOGGER.error(\"Error while performing request\", e);");
            line(content, "          }");
            line(content, "        }");
            line(content, "      });");
            line(content, "      return null;");
            line(content, "    }");
        } else {
            line(content, "    " + clazz + " entry = (" + clazz + ")cache.getOrCreate" + clazz + "(graph, " + signature.keyName + ");");
        }
        line(content, "    " + signature.procedure + " procedure_ = procedure != null ? procedure : emptyProcedure" + clazz + ";");
        line(content, "    if(entry.isReady()) {");
        line(content, "      graph.processor.listening.registerDependencies(graph, entry, parent, listener, procedure_, false);");
        if(genReturn) {
            line(content, "      Object result = entry.performFromCache(graph, procedure_);");
            line(content, "      graph.processor.listening.registerFirstKnown(listener, result);");
            line(content, "      return result;");
        } else {
            line(content, "      entry.performFromCache(graph, procedure_);");
        }
        line(content, "    }");
        line(content, "    else {");
        line(content, "      assert(entry.isPending());");
        if(genReturn) {
            line(content, "      graph.processor.listening.registerDependencies(graph, entry, parent, listener, procedure_, false);");
            if(shortcut) line(content, "      Object result = " + clazz + ".computeForEach(graph, " + signature.keyName + ", entry, procedure_" + (signature.useNeedsToBlock ? ", needsToBlock" : "") + ");");
            else line(content, "      entry.compute(graph, procedure_);"); 
            line(content, "      graph.processor.listening.registerFirstKnown(listener, result);");
            line(content, "      return result;");
        } else {
            line(content, "      graph.processor.listening.registerDependencies(graph, entry, parent, listener, procedure_, false);");
            if(shortcut) line(content, "      " + clazz + ".computeForEach(graph, " + signature.keyName + ", entry, procedure_" + (signature.useNeedsToBlock ? ", needsToBlock" : "") + ");");
            else line(content, "      entry.compute(graph, procedure_);"); 
        }
        line(content, "    }");
        line(content, "}");
        line(content, "");

        String lower = Character.toLowerCase(clazz.charAt(0)) + clazz.substring(1);

        line(content, "private " + clazz + " peek" + clazz + "(" + signature.p1 + ") {");
        line(content, "    synchronized(" + lower +"Map) {");
        line(content, "        return (" + clazz + ") " + lower + "Map.get(" + signature.keyName + ");");
        line(content, "    }");
        line(content, "}");
        line(content, "");
    }

    public void generateRemove(StringBuilder content, String clazz, GenerationInfo signature) {

        String lower = Character.toLowerCase(clazz.charAt(0)) + clazz.substring(1);

        line(content, "void remove(" + clazz + " entry) {");
        line(content, "    synchronized(" + lower + "Map) {");
        line(content, "        " + lower + "Map.remove(entry.id);");
        line(content, "    }");
        line(content, "}");
        line(content, "");

    }

    public void generateGetOrCreate(StringBuilder content, String clazz, GenerationInfo signature) {

        String lower = Character.toLowerCase(clazz.charAt(0)) + clazz.substring(1);

        line(content, "" + clazz + " getOrCreate" + clazz + "(ReadGraphImpl graph, " + signature.p1 + (signature.useNeedsToBlock ? ", boolean needsToBlock" : "") +  ") throws DatabaseException {");
        line(content, "    " + clazz + " existing = null;");
        line(content, "    synchronized(" + lower + "Map) {");
        line(content, "        existing = (" + clazz + ")" + lower + "Map.get(" + signature.keyName + ");");
        line(content, "        if(existing == null) {");
        line(content, "            existing = new " + clazz + "(" + signature.keyName + signature.entryCreationArguments + ");");
        line(content, "            existing.setPending(querySupport);");
        line(content, "            " + lower + "Map.put(" + signature.keyToId + "(" + signature.keyName + "), existing);");
        line(content, "            updates++;");
        line(content, "            size++;");
        line(content, "            return existing;");
        line(content, "        }");
        line(content, "        if(existing.requiresComputation()) {");
        line(content, "            existing.setPending(querySupport);");
        line(content, "            return existing;");
        line(content, "        }");
        line(content, "    }");
        line(content, "    if(existing.isPending()) {");
        if(signature.useNeedsToBlock) {
            line(content, "      if(needsToBlock)");
            line(content, "        waitPending(graph, existing);");
            line(content, "      else {");
            line(content, "        return null;");
            line(content, "      }");
        } else {
            line(content, "      waitPending(graph, existing);");
        }
        line(content, "    }");
        line(content, "    return existing;");
        line(content, "}");
        line(content, "");

    }

    public void generate() {

        URL classLocation = CodeGen.class.getResource(".");
        if (classLocation != null) {
            if (classLocation.getProtocol().equals("file")) {
                try {
                    URL resource = new URL(classLocation, ".");
                    File path = new File(URLDecoder.decode(resource.getPath(), "UTF-8"));
                    String target = path.getAbsolutePath().replace("\\", "/");
                    target = target.replace("/bin/", "/src/") + "/QueryCache.java";
                    System.err.println("target=" + target);
                    File source = new File(target);
                    StringBuilder content = new StringBuilder();
                    content.append("package org.simantics.db.impl.query;\n");
                    content.append("\n");

                    content.append("import org.simantics.db.ObjectResourceIdMap;\n");
                    content.append("import org.simantics.db.RelationInfo;\n");
                    content.append("import org.simantics.db.exception.DatabaseException;\n");
                    content.append("import org.simantics.db.impl.graph.ReadGraphImpl;\n");
                    content.append("import org.simantics.db.impl.procedure.InternalProcedure;\n");
                    content.append("import org.simantics.db.impl.query.QueryProcessor.SessionTask;\n");
                    content.append("import org.simantics.db.procedure.AsyncMultiProcedure;\n");
                    content.append("import org.simantics.db.procedure.AsyncProcedure;\n");
                    content.append("import org.simantics.db.procedure.ListenerBase;\n");
                    content.append("import org.simantics.db.procedure.SyncMultiProcedure;\n");
                    content.append("import org.simantics.db.request.AsyncMultiRead;\n");
                    content.append("import org.simantics.db.request.ExternalRead;\n");
                    content.append("import org.simantics.db.request.MultiRead;\n");
                    content.append("import org.simantics.db.request.Read;\n");
                    content.append("\n");

                    content.append("public class QueryCache extends QueryCacheBase {\n");
                    content.append("\n");

//                    line(content, "private static final boolean SINGLE = true;");
//                    content.append("\n");

                    line(content,"public QueryCache(QuerySupport querySupport, int threads) {");
                    line(content,"    super(querySupport, threads);");
                    line(content,"}");
                    content.append("\n");

                    generateQuery(content, "Objects", signatureR2IP, true, false);
                    generateQuery(content, "Statements", signatureR2TIP, true, false);
                    generateQuery(content, "DirectObjects", signatureR2IP, true, false);
                    generateQuery(content, "RelationInfoQuery", signatureR1RelationInfo, true, false);
                    generateQuery(content, "URIToResource", signatureID1, true, false);
                    generateQuery(content, "ValueQuery", signatureR1Bytes, true, false);
                    generateQuery(content, "OrderedSet", signatureR1IP, true, false);
                    generateQuery(content, "PrincipalTypes", signatureR1IP, true, false);
                    generateQuery(content, "DirectPredicates", signatureR1IntSet, true, false);
                    generateQuery(content, "Predicates", signatureR1IntSet, true, false);
                    generateQuery(content, "ReadEntry", signatureRead, true, true);
                    //generateQuery(content, "AsyncReadEntry", signatureAsyncRead, true, true);
                    generateQuery(content, "Types", signatureR1IntSet, true, false);
                    generateQuery(content, "ChildMap", signatureChildMap, true, false);
                    generateQuery(content, "TypeHierarchy", signatureR1IntSet, true, false);
                    generateQuery(content, "SuperTypes", signatureR1IntSet, true, false);
                    generateQuery(content, "SuperRelations", signatureR1IntSet, true, false);

                    generateQuery(content, "AssertedPredicates", signatureR1IP, false, false);
                    generateQuery(content, "AssertedStatements", signatureR2TIP, false, false);
                    generateQuery(content, "DirectSuperRelations", signatureR1IP, false, false);
                    generateQuery(content, "MultiReadEntry", signatureMultiRead, false, false);
                    generateQuery(content, "AsyncMultiReadEntry", signatureAsyncMultiRead, false, false);
                    generateQuery(content, "ExternalReadEntry", signatureExternalRead, false, false);
                    content.append("}\n");
                    FileUtils.writeFile(source, content.toString().getBytes());
                } catch (MalformedURLException e) {
                    e.printStackTrace();
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }

    }

    public static void main(String[] args) {
        new CodeGen().generate();
    }

}
